package solomon

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"sync"
	"time"

	"a.yandex-team.ru/drive/library/go/auth"
)

// PushClient represents solomon push client.
type PushClient struct {
	config  PushConfig
	closer  chan struct{}
	waiter  sync.WaitGroup
	signals []signal
	mutex   sync.Mutex
	client  http.Client
	auth    auth.Provider
}

// PushConfig configures solomon client.
type PushConfig struct {
	Endpoint string `json:""`
	Project  string `json:""`
	Cluster  string `json:""`
	Service  string `json:""`
}

type signal struct {
	value  interface{}
	time   time.Time
	labels map[string]string
}

const (
	ProductionEndpoint = "https://solomon.yandex.net/api/v2"
)

// NewPushClient creates new push client instance.
func NewPushClient(cfg PushConfig, auth auth.Provider) *PushClient {
	cfg.Endpoint = fixEndpoint(cfg.Endpoint)
	return &PushClient{
		config: cfg,
		client: http.Client{Timeout: time.Minute},
		auth:   auth,
	}
}

// Start starts solomon push client.
//
// You should start client if you want to push your signals to solomon.
func (c *PushClient) Start() {
	if c.closer != nil {
		panic(fmt.Errorf("push client already started"))
	}
	c.closer = make(chan struct{})
	c.waiter.Add(1)
	go c.loop()
}

// Stop stops solomon push client.
//
// You should use this method, when you dont want use client.
// This method also make force push for all sensors.
func (c *PushClient) Stop() {
	if c.closer == nil {
		return
	}
	close(c.closer)
	c.waiter.Wait()
	c.closer = nil
}

type SignalOption func(*signal)

// Label adds label to signal.
func Label(name string, value string) SignalOption {
	return func(s *signal) {
		s.labels[name] = value
	}
}

// Signal appends sensor signal to push queue.
//
// Signal is thread-safe, so it can be called concurrently.
func (c *PushClient) Signal(
	sensor string, value interface{}, options ...SignalOption,
) {
	signal := signal{
		labels: map[string]string{"sensor": sensor},
		value:  value,
		time:   time.Now(),
	}
	for _, option := range options {
		option(&signal)
	}
	c.mutex.Lock()
	defer c.mutex.Unlock()
	c.signals = append(c.signals, signal)
}

func fixEndpoint(endpoint string) string {
	if endpoint == "" || endpoint == "Production" {
		return ProductionEndpoint
	}
	return endpoint
}

func (c *PushClient) loop() {
	defer c.waiter.Done()
	ticker := time.NewTicker(30 * time.Second)
	defer ticker.Stop()
	for {
		select {
		case <-c.closer:
			c.push()
			return
		case <-ticker.C:
			c.push()
		}
	}
}

// TODO(iudovin@): Push signals by batches.
func (c *PushClient) push() {
	events := c.popEvents()
	if len(events) == 0 {
		return
	}
	data, err := buildPayload(events)
	if err != nil {
		c.restoreEvents(events)
		log.Println(err)
		return
	}
	req, err := http.NewRequest(
		http.MethodPost,
		c.config.Endpoint+"/push",
		bytes.NewBuffer(data),
	)
	if err != nil {
		c.restoreEvents(events)
		log.Println(err)
		return
	}
	req.Header.Add("Content-Type", "application/json")
	query := url.Values{}
	query.Set("project", c.config.Project)
	query.Set("cluster", c.config.Cluster)
	query.Set("service", c.config.Service)
	req.URL.RawQuery = query.Encode()
	if err := c.auth.UpdateRequest(req); err != nil {
		c.restoreEvents(events)
		log.Println(err)
		return
	}
	resp, err := c.client.Do(req)
	if err != nil {
		c.restoreEvents(events)
		log.Println("Error:", err)
		return
	}
	defer func() {
		if err := resp.Body.Close(); err != nil {
			log.Println("Error:", err)
		}
	}()
	body, _ := ioutil.ReadAll(resp.Body)
	if resp.StatusCode != http.StatusOK {
		c.restoreEvents(events)
		log.Printf("Status %v with body: %v", resp.Status, string(body))
		return
	}
}

func buildPayload(signals []signal) ([]byte, error) {
	type sensor struct {
		Labels map[string]string `json:"labels"`
		TS     string            `json:"ts"`
		Value  interface{}       `json:"value"`
	}
	var payload struct {
		Sensors []sensor `json:"sensors"`
	}
	for _, signal := range signals {
		ts := signal.time.UTC().Format("2006-01-02T15:04:05.999999999Z")
		payload.Sensors = append(payload.Sensors, sensor{
			Labels: signal.labels,
			Value:  signal.value,
			TS:     ts,
		})
	}
	return json.Marshal(payload)
}

func (c *PushClient) popEvents() []signal {
	c.mutex.Lock()
	defer c.mutex.Unlock()
	events := c.signals
	c.signals = nil
	return events
}

func (c *PushClient) restoreEvents(signals []signal) {
	c.mutex.Lock()
	defer c.mutex.Unlock()
	c.signals = append(c.signals, signals...)
}
