# -*- coding: utf-8 -*-

from requests import Session
from drive.library.py.time import now


__all__ = (
    "SimpleClient",
)


class SimpleClient(object):
    """SimpleClient represents simple client for Solomon API.

    This client supports only TVM2 auth. You should pass ticket_parser
    TVM2 client with specified destination with name "solomon".
    """
    PRODUCTION = "https://solomon.yandex.net/api/v2"

    RETRIES = 3

    def __init__(self, cluster, project, service, tvm2, endpoint=None):
        self._cluster = cluster
        self._project = project
        self._service = service
        self._tvm2 = tvm2
        self._endpoint = endpoint or self.PRODUCTION
        self._client = Session()

    def signal(self, sensor, value, labels=None, time=None):
        payload = self._make_payload([
            self._make_sensor(sensor, value, labels, time),
        ])
        last_exc = None
        for _ in range(self.RETRIES):
            try:
                self._push(payload)
                return
            except Exception as exc:
                last_exc = exc
        raise last_exc

    def _push(self, payload):
        resp = self._client.post(
            "{}/push".format(self._endpoint),
            headers={
                "X-Ya-Service-Ticket":
                    self._tvm2.get_service_ticket_for("solomon"),
            },
            params={
                "cluster": self._cluster,
                "project": self._project,
                "service": self._service,
            },
            json=payload,
        )
        resp.raise_for_status()
        return resp.json()

    @staticmethod
    def _make_sensor(sensor, value, labels=None, time=None):
        time = time or now()
        labels = labels or {}
        return {
            "labels": {"sensor": sensor, **labels},
            "value": value,
            "ts": time.strftime("%Y-%m-%dT%H:%M:%SZ"),
        }

    @staticmethod
    def _make_payload(sensors):
        return {"sensors": sensors}
