# -*- coding: utf-8 -*-

from requests import Session
from .exceptions import TokenExpiredError
from drive.utils.dates import from_string


class BelkacarToken(object):
    TIME_FORMAT = "%Y-%m-%d %H:%M:%S"

    def __init__(self, id, token, refresh_hash, expire_date_time):
        self.id = id
        self.token = token
        self.refresh_hash = refresh_hash
        self.expire_date_time = expire_date_time

    @classmethod
    def unmarshal_dict(cls, data):
        return cls(
            id=data["id"],
            token=data["token"],
            refresh_hash=data["refresh_hash"],
            expire_date_time=from_string(
                data["expire_date_time"], cls.TIME_FORMAT,
            ),
        )

    def marshal_dict(self):
        return dict(
            id=self.id,
            token=self.token,
            refresh_hash=self.refresh_hash,
            expire_date_time=(
                self.expire_date_time.strftime(self.TIME_FORMAT)
            ),
        )


class BelkacarClient(object):
    ENDPOINT = "https://api.belkacar.ru/v2.12-covid19"

    APP_VERSION = "1.24.05"
    USER_AGENT = "okhttp/3.12.3"

    def __init__(self, client=None):
        self._client = client or Session()

    def user(self, device, location, phone_number):
        return self._process_response(self._client.post(
            "{}/user".format(self.ENDPOINT),
            headers={
                "accept": "application/json",
                "user-agent": self.USER_AGENT,
                "-x-belkacar-device-id": device.android_id,
                "-x-belkacar-os": device.os,
                "-x-belkacar-device": device.name,
                "-x-belkacar-user-loc-lng": str(location[0]),
                "-x-belkacar-user-loc-lat": str(location[1]),
                "-x-belkacar-appversion": self.APP_VERSION,
                "-x-belkacar-language": "en",
            },
            data=dict(
                phone=phone_number,
                device_id=device.android_id,
            ),
            timeout=60,
        ))

    def auth_get_code(self, device, location, phone_number):
        return self._process_response(self._client.post(
            "{}/auth/get-code".format(self.ENDPOINT),
            headers={
                "accept": "application/json",
                "user-agent": self.USER_AGENT,
                # "-x-belkacar-device-id": account.device.android_id,
                "-x-belkacar-os": device.os,
                "-x-belkacar-device": device.name,
                "-x-belkacar-user-loc-lng": str(location[0]),
                "-x-belkacar-user-loc-lat": str(location[1]),
                "-x-belkacar-appversion": self.APP_VERSION,
                "-x-belkacar-language": "en",
            },
            data=dict(
                phone=phone_number,
                device_id=device.android_id,
            ),
            timeout=60,
        ))

    def auth(
        self, device, location, phone_number,
        confirmation_code, confirmation_hash,
    ):
        data = self._process_response(self._client.post(
            "{}/auth".format(self.ENDPOINT),
            headers={
                "accept": "application/json",
                "user-agent": self.USER_AGENT,
                # "-x-belkacar-device-id": account.device.android_id,
                "-x-belkacar-os": device.os,
                "-x-belkacar-device": device.name,
                "-x-belkacar-user-loc-lng": str(location[0]),
                "-x-belkacar-user-loc-lat": str(location[1]),
                "-x-belkacar-appversion": self.APP_VERSION,
                "-x-belkacar-language": "en",
            },
            data=dict(
                phone=phone_number,
                device_id=device.android_id,
                auth_method="sms",
                confirmation_code=confirmation_code,
                confirmation_hash=confirmation_hash,
            ),
            timeout=60,
        ))
        return BelkacarToken.unmarshal_dict(data)

    def get_cars(self, device, location, token):
        return self._process_response(self._client.get(
            "{}/car".format(self.ENDPOINT),
            headers={
                "accept": "application/json",
                "user-agent": self.USER_AGENT,
                "-x-belkacar-auth-token": token.token,
                "-x-belkacar-device-id": device.android_id,
                "-x-belkacar-os": device.os,
                "-x-belkacar-device": device.name,
                "-x-belkacar-user-loc-lng": str(location[0]),
                "-x-belkacar-user-loc-lat": str(location[1]),
                "-x-belkacar-appversion": self.APP_VERSION,
                "-x-belkacar-language": "en",
            },
            params={
                "longitude": str(location[0]),
                "latitude": str(location[1]),
                # "radius": 0, "limit": 0, "offset": 0,
                # Magic constants
                "type_id_list[0]": "1",
                "type_id_list[1]": "2",
                "type_id_list[2]": "3",
                "type_id_list[3]": "4",
                "type_id_list[4]": "5",
                "type_id_list[5]": "38",
                "type_id_list[6]": "39",
            },
            timeout=60,
        ))

    def get_car_info(self, device, location, token, car_id):
        return self._process_response(self._client.get(
            "{}/car-offer".format(self.ENDPOINT),
            headers={
                "accept": "application/json",
                "user-agent": self.USER_AGENT,
                "-x-belkacar-auth-token": token.token,
                "-x-belkacar-device-id": device.android_id,
                "-x-belkacar-os": device.os,
                "-x-belkacar-device": device.name,
                "-x-belkacar-user-loc-lng": str(location[0]),
                "-x-belkacar-user-loc-lat": str(location[1]),
                "-x-belkacar-appversion": self.APP_VERSION,
                "-x-belkacar-language": "en",
            },
            params={
                "longitude": str(location[0]),
                "latitude": str(location[1]),
                "car_id": str(car_id),
            },
            timeout=60,
        ))

    def car_types(self, device, location, token):
        return self._process_response(self._client.get(
            "{}/car-types".format(self.ENDPOINT),
            headers={
                "accept": "application/json",
                "user-agent": self.USER_AGENT,
                "-x-belkacar-auth-token": token.token,
                "-x-belkacar-device-id": device.android_id,
                "-x-belkacar-os": device.os,
                "-x-belkacar-device": device.name,
                "-x-belkacar-user-loc-lng": str(location[0]),
                "-x-belkacar-user-loc-lat": str(location[1]),
                "-x-belkacar-appversion": self.APP_VERSION,
                "-x-belkacar-language": "en",
            },
            params={
                "longitude": str(location[0]),
                "latitude": str(location[1]),
            },
            timeout=60,
        ))

    def fare(self, device, location, token):
        return self._process_response(self._client.get(
            "{}/car-types".format(self.ENDPOINT),
            headers={
                "accept": "application/json",
                "user-agent": self.USER_AGENT,
                "-x-belkacar-auth-token": token.token,
                "-x-belkacar-device-id": device.android_id,
                "-x-belkacar-os": device.os,
                "-x-belkacar-device": device.name,
                "-x-belkacar-user-loc-lng": str(location[0]),
                "-x-belkacar-user-loc-lat": str(location[1]),
                "-x-belkacar-appversion": self.APP_VERSION,
                "-x-belkacar-language": "en",
            },
            params={
                "longitude": str(location[0]),
                "latitude": str(location[1]),
                # Magic constants
                "description": "1",
            },
            timeout=60,
        ))

    def refresh_token(self, device, location, phone_number, token):
        data = self._process_response(self._client.post(
            "{}/auth/refresh-token".format(self.ENDPOINT),
            headers={
                "accept": "application/json",
                "user-agent": self.USER_AGENT,
                "-x-belkacar-device-id": device.android_id,
                "-x-belkacar-os": device.os,
                "-x-belkacar-device": device.name,
                "-x-belkacar-user-loc-lng": str(location[0]),
                "-x-belkacar-user-loc-lat": str(location[1]),
                "-x-belkacar-appversion": self.APP_VERSION,
                "-x-belkacar-language": "en",
            },
            data=dict(
                phone=phone_number,
                device_id=device.android_id,
                token=token.token,
                refresh_hash=token.refresh_hash,
            ),
            timeout=60,
        ))
        return BelkacarToken.unmarshal_dict(data)

    @staticmethod
    def _process_response(resp):
        resp.raise_for_status()
        data = resp.json()
        if "error_code" in data:
            if data["error_code"] == 460:
                # result["error_text"] = "TOKEN IS EXPIRED"
                raise TokenExpiredError()
            raise RuntimeError(
                "Belkacar API returned error '{}': {}".
                format(data["error_code"], data.get("error_text"))
            )
        return data
