# -*- coding: utf-8 -*-

from hashlib import md5
from time import time
from uuid import uuid4
from requests import Session

from drive.library.py.time import TZ_MSK, to_timezone, now


class DelimobilClient(object):
    ENDPOINT = "https://api.delitime.ru/api"

    APP_VERSION = "7.19.0, build 0225c1a79"
    USER_AGENT = "okhttp/4.3.1"
    DEVICE_TYPE = "3"

    def __init__(self, client=None):
        self._client = client or Session()

    def signup(self, device, phone):
        return self._process_response(self._client.post(
            self.ENDPOINT + "/v2/signup",
            headers=self._get_headers(device, None),
            data={
                "SignupForm[username]": str(phone),
                "SignupForm[device_token]": device.android_id,
                "SignupForm[device_type]": self.DEVICE_TYPE,
            },
        ))

    def login(self, device, phone, code):
        return self._process_response(self._client.post(
            self.ENDPOINT + "/v2/auth",
            headers=self._get_headers(device, None),
            data={
                "LoginForm[username]": str(phone),
                "LoginForm[password]": md5(str(code).encode("utf-8")).hexdigest(),
                "LoginForm[device_token]": device.android_id,
                "LoginForm[device_type]": self.DEVICE_TYPE,
            },
        ))

    def cars(self, device, token):
        return self._process_response(self._client.get(
            self.ENDPOINT + "/v4/cars",
            headers=self._get_headers(device, token),
            params={
                "with": "fuel",
            },
        ))

    def car_models(self, token):
        return self.__process_response(self.session.get(
            self.ENDPOINT + "/v3/cars/models",
            headers=self.__get_headers(token),
        ))

    def car_info(self, car_id, token):
        return self.__process_response(self.session.get(
            self.ENDPOINT + "/v2/car-info",
            headers=self.__get_headers(token),
            params={
                "id": str(car_id),
            },
        ))

    def car_rent_info(self, token):
        return self.__process_response(self.session.get(
            self.ENDPOINT + "/v1/car-rent-info",
            headers=self.__get_headers(token),
        ))

    def car_reserve(self, car_id, contract_id, token, rent_type="minutes"):
        return self.__process_response(self.session.get(
            self.ENDPOINT + "/v3/car-reserve",
            headers=self.__get_headers(token),
            params={
                "id": str(car_id),
                "rent_type": rent_type,
                "contract_id": str(contract_id),
                "time": to_timezone(now(), TZ_MSK).strftime("%Y%m%d%H%M%S"),
            },
        ))

    def regions(self, device, token):
        return self._process_response(self._client.get(
            self.ENDPOINT + "/v1/regions",
            headers=self._get_headers(device, token),
        ))

    def tariff(self, device, token):
        return self._process_response(self._client.get(
            self.ENDPOINT + "/v1/tariff",
            headers=self._get_headers(device, token),
        ))

    def change_region(self, device, region, token):
        return self._process_response(self._client.post(
            self.ENDPOINT + "/v1/change-region",
            headers=self._get_headers(device, token),
            data={
                "ChangeRegionForm[region_id]": region,
            }
        ))

    def countries(self, device):
        return self._process_response(self._client.post(
            self.ENDPOINT + "/v2/countries",
            headers=self._get_headers(device, None),
            data={
                "with": "allowed",
            }
        ))

    def _get_headers(self, device, token):
        headers = {
            "Accept-Language": "en",
            "timestamp": str(int(round(time() * 1000))),
            "version": self.APP_VERSION,
            "AID": device.android_id,
            "UUID": str(uuid4()),
            "model": device.name,
            "Accept-Encoding": "gzip",
            "User-Agent": self.USER_AGENT,
            # "If-None-Match": "W/\"6221e-npbd+XveY+zqynZiwmsNUcyebg0\""
        }
        if token is not None:
            headers["Authorization"] = "Bearer {}".format(token)
        return headers

    @staticmethod
    def _process_response(resp):
        resp.raise_for_status()
        return resp.json()


class DelimobilV2:
    ENDPOINT = "https://api.delimobil.ru/api"

    def __init__(self, client):
        self._client = client or Session()

    def get_geozones(self):
        resp = self._client.get(
            self.ENDPOINT + "/geozones",
            headers=self._get_headers(),
        )
        resp.raise_for_status()
        return resp.json()

    def get_cars(self, region_id):
        resp = self._client.get(
            self.ENDPOINT + "/cars",
            params={"regionId": str(region_id)},
            headers=self._get_headers(),
        )
        resp.raise_for_status()
        return resp.json()

    def _get_headers(self):
        return {
            "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/42.0.2311.135 Safari/537.36 Edge/12.246"
        }
