# -*- coding: utf-8 -*-

from datetime import datetime
from dateutil.tz import tzutc, tzoffset


TZ_UTC = tzutc()
TZ_MSK = tzoffset("MSK", 10800)

EPOCH = datetime(1970, 1, 1, tzinfo=TZ_UTC)


def now():
    """now returns current time in UTC timezone.
    """
    return datetime.utcnow().replace(tzinfo=TZ_UTC)


def from_timestamp(ts):
    """from_timestamp converts amount of seconds from unix epoch to datetime.
    """
    return datetime.utcfromtimestamp(ts).replace(tzinfo=TZ_UTC)


def from_string(s, fmt):
    """from_string converts string in specified format to datetime.
    """
    dt = datetime.strptime(s, fmt)
    if dt.tzinfo is None:
        dt = dt.replace(tzinfo=TZ_UTC)
    return dt


def to_timestamp(dt):
    """to_timestamp converts datetime to amount of seconds from unix epoch.

    If timezone is not present, we will assume that datetime in UTC.

    Returns float value.
    """
    if dt.tzinfo is None:
        dt = dt.replace(tzinfo=TZ_UTC)
    return (dt - EPOCH).total_seconds()


def to_timezone(dt, tz):
    """to_timezone converts datetime to another timezone.

    Should be always true:
      to_timestamp(dt) == to_timestamp(to_timezone(dt, tz))
    """
    return dt.astimezone(tz)
