#include "uuid_meta_cache.h"

#include <drive/pq2saas/libprobes/caches.h>
#include <zootopia/library/cc/libtimedcache/timed_cache.h>

#include <library/cpp/deprecated/atomic/atomic.h>


namespace NPq2Saas {

LWTRACE_USING(PQ2SAAS_CACHES_PROVIDER);

class TUUIDMetaCache::TImpl {
public:
    TImpl(const TDuration& timeTillOutdated, const TDuration& timeTillTryUpdate,
          const TDuration& updateNotFasterThen, int maxWaitAttempts, TDuration sleepBetweenAttempts)
        : CacheSize(0)
        , Cache(timeTillOutdated, timeTillTryUpdate, updateNotFasterThen)
        , MaxWaitAttempts(maxWaitAttempts)
        , SleepBetweenAttempts(sleepBetweenAttempts)
    {}

    bool Get(const TString& key, bool& value) const {
        LWPROBE(UUIDMetaCacheRead, key);
        auto result = Cache.Get(key, value, MaxWaitAttempts, SleepBetweenAttempts, false);
        if (result) {
            LWPROBE(UUIDMetaCacheReadHit, key, value);
        } else {
            LWPROBE(UUIDMetaCacheReadMiss, key, value);
        }
        return result;
    }

    void Set(const TString& key, const bool& value) {
        LWPROBE(UUIDMetaCacheWriteBegin, key, value, Size());
        bool isNewKey = Cache.Set(key, value);
        if (isNewKey) {
            AtomicIncrement(CacheSize);
        }
        LWPROBE(UUIDMetaCacheWriteEnd, key, Size());
    }

    ui64 Size() const {
        return AtomicGet(CacheSize);
    }

private:
    volatile ui64 CacheSize;
    NTimedCache::TTimedCache<TString, bool, 1024> Cache;
    const int MaxWaitAttempts;
    const TDuration SleepBetweenAttempts;
};

TUUIDMetaCache::TUUIDMetaCache(const TDuration& timeTillOutdated,
                               const TDuration& timeTillTryUpdate,
                               const TDuration& updateNotFasterThen,
                               int maxWaitAttempts,
                               TDuration sleepBetweenAttempts)
    : Impl(new TImpl{timeTillOutdated, timeTillTryUpdate, updateNotFasterThen, maxWaitAttempts, sleepBetweenAttempts})
{}

TUUIDMetaCache::~TUUIDMetaCache() = default;

bool TUUIDMetaCache::Get(const TString& uuid, bool& value) const {
    return Impl->Get(uuid, value);
}

void TUUIDMetaCache::Set(const TString& uuid, const bool& value) {
    Impl->Set(uuid, value);
}

ui64 TUUIDMetaCache::Size() const {
    return Impl->Size();
}

} // namespace NPq2Saas
