#include "config.h"

#include <library/cpp/archive/yarchive.h>
#include <library/cpp/protobuf/json/json2proto.h>

#include <util/generic/singleton.h>
#include <util/generic/string.h>
#include <util/memory/blob.h>
#include <util/stream/file.h>
#include <util/stream/str.h>
#include <util/string/printf.h>


namespace {

constexpr size_t CONFIG_SINGLETON_PRIORITY = 100;

const unsigned char CONFIGS_DATA[] = {
#include "configs"
};

class TConfigLoader : public TArchiveReader {
public:
    TConfigLoader()
        : TArchiveReader(TBlob::NoCopy(CONFIGS_DATA, sizeof(CONFIGS_DATA)))
    {
    }
};

/// @throw yexception
TString LoadPresetData(const TString& configName) {
    const TString configFileName = Sprintf("/pq2saas_%s.json", configName.data());
    if (!Singleton<TConfigLoader>()->Has(configFileName)) {
        throw yexception() << Sprintf("Unknown config '%s'.", configName.data());
    }
    return Singleton<TConfigLoader>()->ObjectByKey(configFileName)->ReadAll();

}

NPq2SaasProto::TConfig LoadConfig(const TString& path, bool loadFromPresets) {
    auto data = loadFromPresets ? LoadPresetData(path) : TUnbufferedFileInput(path).ReadAll();
    return NProtobufJson::Json2Proto<NPq2SaasProto::TConfig>(data);
}

} // anonymous namespace

namespace NPq2Saas {

void InitConfig(const TString& pathToConfig, bool lookInPresets) {
    Cerr << "Loading '" << pathToConfig << "' config...";
    if (lookInPresets) {
        Cerr << " Using presets for lookup.";
    }
    Cerr << Endl;
    auto config = LoadConfig(pathToConfig, lookInPresets);
    *SingletonWithPriority<TConfig, CONFIG_SINGLETON_PRIORITY>() = config;
}

const TConfig& GetConfig() {
    return *SingletonWithPriority<TConfig, CONFIG_SINGLETON_PRIORITY>();
}

} // namespace NPq2Saas
