#include "consumer.h"

#include <drive/pq2saas/libhandling/factory.h>
#include <drive/pq2saas/libhandling/interface/interface.h>
#include <drive/pq2saas/liblogging/log.h>
#include <drive/pq2saas/libprobes/consumer.h>

#include <zootopia/library/cc/libtskv/tskv_2_dict_parser.h>

#include <rtline/library/storage/abstract.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/unistat/unistat.h>

#include <util/datetime/base.h>
#include <util/generic/algorithm.h>
#include <util/generic/fwd.h>
#include <util/generic/yexception.h>
#include <util/system/tls.h>

namespace {

using namespace NPq2Saas;

THashSet<TString> GetAllAcceptedEventNames(const TCallbackSettings& settings) {
    THashSet<TString> result;
    for (const auto& kvPair : settings.Deliveries) {
        if (const auto ptr = kvPair.second.GetInterfacePtr()) {
            for (const auto& eventName : ptr->Filtering.GetAcceptedEventNames()) {
                result.insert(eventName);
            }
        }
    }
    return result;
}

Y_POD_STATIC_THREAD(ui64) IterationTrackId = 0;
Y_POD_STATIC_THREAD(TInstant) LastHardCommitTime;

void UpdateIterationTrackId() {
    IterationTrackId = Now().MicroSeconds();
}

} // anonymous namespace

namespace NPq2Saas {

LWTRACE_USING(PQ2SAAS_CONSUMER_PROVIDER);
using namespace NPersQueue2;

TConsumerCallback::TConsumerCallback(const TCallbackSettings& settings,
                                     const TString& topic,
                                     TConsumerRecId position,
                                     TMaybe<TFuture<void>> isDead,
                                     TAtomic& isSoftCommitEnabled,
                                     TAtomic& isStopRequested,
                                     TAtomicSharedPtr<TThreadPool> parsingQueue,
                                     TAtomicSharedPtr<TThreadPool> sendingQueue,
                                     NPq2SaasMonitoring::TManager& monManager,
                                     TSyncFactory syncFactory
)
        : NPersQueue2::ICallBack()
        , Settings(settings)
        , Topic(topic)
        , Position(position)
        , IsDead(isDead)
        , IsSoftCommitEnabled(isSoftCommitEnabled)
        , IsStopRequested(isStopRequested)
        , ParsingQueue(parsingQueue)
        , SendingQueue(sendingQueue)
        , MonManager(monManager)
        , SyncFactory(syncFactory)
{
    UpdateIterationTrackId();
    LastHardCommitTime = TInstant::Now();
    Sync = SyncFactory(Topic);
}

TConsumerCallback::~TConsumerCallback() {
    // Callback produces tasks, that can capture it, so we wait for them.
    while (ActiveTasks.Val() > 0) {
        Sleep(TDuration::MilliSeconds(100));
    }
    LWPROBE(ThreadDown, Topic);
    MonManager.LockedTopics->Dec();

    LOG_JSON.KV("method", "TConsumerCallback::~TConsumerCallback")
            .KV("json_message", Sprintf("\"TopicLock: %s\"", Topic.data()));
}

ui64 TConsumerCallback::NextIteration() {
    UpdateIterationTrackId();
    return IterationTrackId;
}

void TConsumerCallback::WaitForSender() const {
    LWPROBE(WaitForSender_Begin, IterationTrackId, Topic, Position, ParsingQueue->Size(), SendingQueue->Size());
    while (Settings.MaxInFlight && SendingQueue->Size() > Settings.MaxInFlight * 0.5) {
        Sleep(Settings.WaitSenderTimeout);
    }
    LWPROBE(WaitForSender_End, IterationTrackId, Topic, Position, ParsingQueue->Size(), SendingQueue->Size());
}

TConsumerRecId TConsumerCallback::BeforeRead(bool* softCommit) {
    Y_VERIFY(!!Topic, "Using an uninitialized callback");
    Y_VERIFY(!!IsDead, "Using an uninitialized callback");

    UpdateIterationTrackId();

    while (true) {
        TMaybe<TConsumerRecId> pos = Sync->GetPosition(TDuration::Seconds(1));

        if (AtomicGet(IsStopRequested) || IsDead->HasValue()) {
            break;
        }
        if (pos.Empty()) {
            continue;
        }

        if (pos.GetRef() != NO_POSITION_CHANGE) {
            // Got position from storage, hard commit it
            Position = pos.GetRef();
            *softCommit = false;

            LOG_JSON.KV("method", "TConsumerCallback::BeforeRead")
                    .KV("json_message", Sprintf("\"CommitNewPosition: %ld\", Topic: %s", Position, Topic.data()));
        }
        break;
    }

    LWPROBE(BeforeRead_Begin, IterationTrackId, Topic, Position, ParsingQueue->Size(), SendingQueue->Size());
    if (Settings.MaxInFlight > 0 && SendingQueue->Size() > Settings.MaxInFlight) {
        MonManager.BlockedConsumerThreads->Inc();
        WaitForSender();
        MonManager.BlockedConsumerThreads->Dec();
    }
    LWPROBE(BeforeRead_End, IterationTrackId, Topic, Position, ParsingQueue->Size(), SendingQueue->Size());
    return Position;
}

TConsumerRecId TConsumerCallback::AfterRead(TConsumerRecId nextPos, bool* softCommit) {
    LWPROBE(AfterRead, IterationTrackId, Topic, nextPos, ParsingQueue->Size(), SendingQueue->Size());
    return ICallBack::AfterRead(nextPos, softCommit);
}

TConsumerRecId TConsumerCallback::OnData(NPersQueue2::TBlob* blob, const TProducerInfo& info,
                                         size_t size, TConsumerRecId nextPos, bool* softCommit) {
    Y_UNUSED(info);
    TInstant receivedTime = TInstant::Now();
    *softCommit = (AtomicGet(IsSoftCommitEnabled) == 1);

    LWPROBE(OnData_Begin, IterationTrackId, Topic, size, nextPos, ParsingQueue->Size(), SendingQueue->Size());
    for (size_t i = 0; i < size; ++i) {
        MonManager.ReportBlobSize(blob[i].size());
        ParseBlob(blob[i], receivedTime);
        if (blob[i].size() > Settings.MaxReservedBufferKbSize) {
            blob[i].Resize(Settings.MaxReservedBufferKbSize);
            blob[i].ShrinkToFit();
        }
    }
    Position = nextPos;

    if (*softCommit && (LastHardCommitTime + Settings.HardCommitPeriod < TInstant::Now())) {
        *softCommit = false;
        LastHardCommitTime = TInstant::Now();
        // save current position to persistent storage
        Sync->SetPosition(Position);
    }

    LWPROBE(OnData_End, IterationTrackId, Topic, size, Position,
            ParsingQueue->Size(), SendingQueue->Size(), *softCommit, LastHardCommitTime.Seconds());
    return Position;
}

TAtomicSharedPtr<NPersQueue2::ICallBack>
TConsumerCallback::Clone(const TString& topic, TConsumerRecId offset, TConsumerRecId lag, TFuture<void>&& isDead) {
    LWPROBE(ThreadUp, topic, offset, lag);
    auto position = Settings.SkipInitialLag ? (offset + lag) : offset;
    MonManager.LockedTopics->Inc();

    LOG_JSON.KV("method", "TConsumerCallback::Clone")
            .KV("json_message", Sprintf("\"TopicLock: %s\"", topic.data()));

    return MakeAtomicShared<TConsumerCallback>(Settings, topic, position, MakeMaybe(isDead),
                                               IsSoftCommitEnabled, IsStopRequested,
                                               ParsingQueue, SendingQueue,
                                               MonManager, SyncFactory);
}

void TConsumerCallback::ParseBlob(NPersQueue2::TBlob& origBlobData, TInstant receivedTime) {
    ActiveTasks.Inc();
    (void)ParsingQueue->AddFunc([blobData(std::move(origBlobData)), receivedTime, this]{
        TStringBuf blob(blobData.Data(), blobData.Size());
        TStringBuf line;
        while (blob.NextTok('\n', line)) try {
            THashMap<TString, TString> data;
            switch (Settings.Format) {
            case TCallbackSettings::EFormat::Tskv:
                data = NTskvBuilder::Tskv2Dict(line);
                break;
            case TCallbackSettings::EFormat::Json: {
                NJson::TJsonValue json;
                NJson::ReadJsonFastTree(line, &json, true);
                for (auto&& i : json.GetMapSafe()) {
                    data[i.first] = i.second.GetStringRobust();
                }
                break;
            }
            default:
                Y_FAIL("unimplemented");
                break;
            }

            for (const auto& kvPair : Settings.Deliveries) {
                const auto& deliveryName = kvPair.first;
                const auto& handlerSettings = kvPair.second;
                const auto handlerType = handlerSettings.GetInterfacePtr()->HandlerType;
                auto handler = CreateEventHandler(handlerType, deliveryName,
                                                  handlerSettings,
                                                  MonManager, Settings.Manager,
                                                  SendingQueue.Get());
                if (handler) {
                    auto handlerName = ToString(handlerType);
                    try {
                        handler->OnEvent(data);
                        MonManager.GetDeliveryStats(deliveryName)->EventProcessingTime.ReportEvent(receivedTime);
                        TUnistat::Instance().PushSignalUnsafe(deliveryName + "-success", 1);
                    } catch (const IEventHandler::TBadInputException& e) {
                        LWPROBE(BadHandlerInput, handlerName, TString{line}, FormatExc(e));
                        MonManager.GetBadHandlerInputCounter(handlerName)->Inc();
                        TUnistat::Instance().PushSignalUnsafe(deliveryName + "-failure", 1);
                    } catch (const std::exception& e) {
                        LWPROBE(HandlerFail, handlerName, TString{line}, FormatExc(e));
                        MonManager.GetHandlerFailCounter(handlerName)->Inc();
                        TUnistat::Instance().PushSignalUnsafe(deliveryName + "-failure", 1);
                    }
                } else {
                    MonManager.UnsupportedLogTypeMessages->Inc();
                }
            }
        } catch (const std::exception& e) {
            ERROR_LOG << "an exception occurred while parsing line '" << line << "': " << FormatExc(e) << Endl;
        }
        ActiveTasks.Dec();
    });
}


TConsumerCallbackManager::TConsumerCallbackManager(const TCallbackSettings& settings)
        : Settings(settings)
        , IsSoftCommitEnabled(1)
        , IsStopRequested(0)
        , ParsingQueue(MakeAtomicShared<TThreadPool>(TThreadPool::TParams().SetBlocking(true).SetCatching(true)))
        , SendingQueue(MakeAtomicShared<TThreadPool>(TThreadPool::TParams().SetBlocking(false).SetCatching(true)))
        , MonManager(GetAllAcceptedEventNames(Settings))
{
    Cerr << "== Supported handlers: ==" << Endl;
    for (const auto& handlerName : GetEventHandlersNames()) {
        Cerr << "  " << handlerName << Endl;
    }
    for (const auto& kvPair : Settings.Deliveries) {
        auto handlerType = kvPair.second.GetInterfacePtr()->HandlerType;
        auto handlerTypeStr = ToString(handlerType);
        if (!HasEventHandler(handlerType)) {
            throw yexception() << "No matching handler found for event type: "
                               << GetHandlerTypeName(handlerType);
        }
        auto deliveryStats = MonManager.AddDeliveryStats(kvPair.first);
        for (const auto& dependency : kvPair.second.GetInterfacePtr()->GetDependencies()) {
            if (dependency.Type != THandlerDependency::Cache) {
                deliveryStats->AddPerDependencyStats(dependency.Name);
            }
        }
        MonManager.AddHandlerFailCounter(handlerTypeStr);
        MonManager.AddBadHandlerInputCounter(handlerTypeStr);
        TUnistat::Instance().DrillFloatHole(kvPair.first + "-success", "dmmm", NUnistat::TPriority(50));
        TUnistat::Instance().DrillFloatHole(kvPair.first + "-failure", "dmmm", NUnistat::TPriority(50));
        kvPair.second.GetInterfacePtr()->RegisterUnistatSignalHoles(kvPair.first);
    }
    ParsingQueue->Start(Settings.ParsingWorkersCount, Settings.MaxParsingQueueSize);
    SendingQueue->Start(Settings.WorkersCount);
}

THolder<TConsumerCallback> TConsumerCallbackManager::CreateCallback() {
    auto syncFactory = [this] (const TString& topic) {
        return CreatePartitionSync(topic);
    };
    return MakeHolder<TConsumerCallback>(Settings, "", -1, Nothing(), IsSoftCommitEnabled, IsStopRequested,
                                         ParsingQueue, SendingQueue, MonManager, syncFactory);
}

void TConsumerCallbackManager::RequestStop() {
    AtomicSet(IsStopRequested, 1);
}

void TConsumerCallbackManager::Wait() {
    Y_VERIFY(AtomicGet(IsStopRequested), "Stop() is not requested");

    AtomicSet(IsSoftCommitEnabled, 0);
    ParsingQueue->Stop();
    SendingQueue->Stop();
}

IPartitionSync* TConsumerCallbackManager::CreatePartitionSync(const TString& topic) {
    Y_UNUSED(topic);
    return new TEmptyPartitionSync();
}


} // namespace NPq2Saas
