#include "sync.h"

#include <util/string/split.h>


namespace {

TString GetDC(const TString& topic) {
    // Topic looks like this:
    // rt3.fol--maps_analyzer_tracks--analyzer-tracks-log:0
    return TString{TStringBuf(topic).After('.').Before('-')};
}

} // anonymous namespace

namespace NPq2Saas {

using namespace NPersQueue2;

TStoragePartitionSync::TStoragePartitionSync(const TString& topic,
                                             const TCallbackSettings& settings,
                                             NRTLine::IVersionedStorage::TPtr storage,
                                             NPq2SaasMonitoring::TManager& monManager)
    : Topic(topic)
    , Settings(settings)
    , Storage(storage)
    , MonManager(monManager)
    , LockNode(topic + "/lock")
    , PositionNode(topic + "/position")
    , PrimaryIsWaitingNode(topic + "/primary_is_here")
{
}

TStoragePartitionSync::~TStoragePartitionSync() {
    if (Lock) {
        // lock has been acquired once, but hasn't been released
        LockReleased();
    }
}

TMaybe<TConsumerRecId> TStoragePartitionSync::GetPosition(TDuration timeout) {
    // check if primary reader asking us to give up this partition
    if (ShouldGiveUpLock()) {
        // release the lock if it's acquired
        if (Lock) {
            Lock = nullptr;
            LockReleased();
        }
        // sleep a bit to give primary a good chance
        Sleep(Min(timeout, TDuration::Seconds(1)));
        return Nothing();
    }
    // check if partition is already locked
    if (Lock) {
        if (Lock->IsLocked()) {
            return NO_POSITION_CHANGE;
        } else {
            // lock has been lost
            Lock = nullptr;
            LockReleased();
        }
    }
    // try acquire lock
    if (Lock = Storage->WriteLockNode(LockNode, timeout)) {
        // got the lock, discard wait token
        if (IsPrimary()) {
            SetPrimaryIsWaiting(false);
        }
        LockAcquired();
        // read position from storage
        return GetPositionFromStorage();
    }
    // acquire lock failed, probably someone else is holding it
    if (IsPrimary()) {
        SetPrimaryIsWaiting(true);
    }
    return Nothing();
}

void TStoragePartitionSync::SetPosition(TConsumerRecId position) {
    Storage->SetValue(PositionNode, ToString<TConsumerRecId>(position),
                      /* storeHistory = */ false, /* lock = */ false);
}

void TStoragePartitionSync::LockAcquired() {
    MonManager.ExclusiveReadLockedTopics->Inc();
}

void TStoragePartitionSync::LockReleased() {
    MonManager.ExclusiveReadLockedTopics->Dec();
}

bool TStoragePartitionSync::IsPrimary() const {
    const TString& clientDataCenter = Settings.DataCenter;
    const TString& topicDataCenter = GetDC(Topic);
    return
        clientDataCenter == topicDataCenter ||
        Settings.DataCenterAffinity.contains(std::make_pair(clientDataCenter, topicDataCenter));
}

bool TStoragePartitionSync::ShouldGiveUpLock() const {
    if (IsPrimary() || !Storage->ExistsNode(PrimaryIsWaitingNode)) {
        return false;
    }

    TString result;
    Storage->GetValue(PrimaryIsWaitingNode, result, /* version = */ -1, /* lock = */ true);

    if (result) {
        return TInstant::Now() - TInstant::ParseIso8601Deprecated(result) < TDuration::Seconds(32);
    } else {
        return false;
    }
}

void TStoragePartitionSync::SetPrimaryIsWaiting(bool value) {
    if (value) {
        Storage->SetValue(PrimaryIsWaitingNode, TInstant::Now().ToString(),
                          /* storeHistory = */ false, /* lock = */ true);
    } else {
        auto lock = Storage->WriteLockNode(PrimaryIsWaitingNode);
        Storage->RemoveNode(PrimaryIsWaitingNode, /* with_history = */ false);
    }
}

TMaybe<TConsumerRecId> TStoragePartitionSync::GetPositionFromStorage() const {
    if (!Storage->ExistsNode(PositionNode)) {
        return NO_POSITION_CHANGE;
    }

    TString result;
    Storage->GetValue(PositionNode, result, /* version = */ -1, /* lock = */ true);

    if (result) {
        return FromString<TConsumerRecId>(result);
    } else {
        return Nothing();
    }
}


} // namespace NPq2Saas
