#include "event.h"

#include <rtline/api/action.h>

#include <library/cpp/digest/sfh/sfh.h>
#include <library/cpp/json/json_value.h>

#include <util/digest/city.h>
#include <util/string/cast.h>
#include <util/string/hex.h>
#include <util/string/printf.h>

#include <cmath>
#include <limits>


namespace {
inline bool IsZeroDouble(double val) {
    return std::abs(val) < std::numeric_limits<double>::epsilon();
}
} // anonymous namespace


namespace NPq2Saas {

TBaseSaasEvent::TBaseSaasEvent(const TString& uid)
    : UID(uid)
{}

void TBaseSaasEvent::SetTS(const TString& ts) {
    TS = FromStringWithDefault<double>(ts);
}

double TBaseSaasEvent::GetTS() const {
    return TS;
}

const TString& TBaseSaasEvent::GetUID() const {
    return UID;
}

ui32 TBaseSaasEvent::GetUidHash() const {
    return SuperFastHash(UID.data(), UID.size());
}


TLocationEvent::TLocationEvent(const TString& uid)
    : TBaseSaasEvent(uid)
    , GeoShardingEnabled(false)
{}

void TLocationEvent::SetPosition(double lat, double lon) {
    Lat = lon;
    Lon = lat;
}

bool TLocationEvent::HasValidPosition() const {
    return !(IsZeroDouble(Lat) && IsZeroDouble(Lon));
}

void TLocationEvent::EnableGeoSharding() {
    GeoShardingEnabled = true;
}

NRTLine::TAction TLocationEvent::ToSaasAction() const {
    NRTLine::TAction action;
    NRTLine::TDocument& doc = action.GetDocument();
    doc.SetUrl(UID);
    doc.AddProperty("TS", Sprintf("%.3f", TS));
    Y_VERIFY(GeoShardingEnabled);
    if (GeoShardingEnabled) {
        doc.AddGeoData().AddCoordinate({Lat, Lon}).SetType("TRAFFIC_LOCATION");
    }
    return action;
}


TMetrikaEvent::TMetrikaEvent(const TString& uid, const TString& did)
    : TLocationEvent(uid)
    , DID(did)
{}

void TMetrikaEvent::SetAccountID(const TString& accountID) {
    AccountID = accountID;
}

void TMetrikaEvent::SetRouteUniqueID(const TString& routeUniqueID) {
    RouteUniqueID = routeUniqueID;
}

void TMetrikaEvent::SetType(const TString& eventName) {
    EventType = eventName;
}

void TMetrikaEvent::SetEventNumber(ui64 eventNumber) {
    EventNumber = ToString(eventNumber);
}

void TMetrikaEvent::SetExtProperties(const NJson::TJsonValue& json) {
    ExtProperties.clear();
    ExtProperties.reserve(json.GetMap().size());
    for (const auto& kvPair : json.GetMap()) {
        ExtProperties.push_back({kvPair.first, kvPair.second.GetString()});
    }
}

NRTLine::TAction TMetrikaEvent::ToSaasAction() const {
    NRTLine::TAction action;
    NRTLine::TDocument& doc = action.GetDocument();
    doc.SetUrl(UID);
    doc.AddSearchAttribute("s_device_id", DID);
    if (AccountID) {
        doc.AddProperty("ACCID", AccountID);
    }
    if (EventType) {
        doc.AddProperty("EventType", EventType);
    }
    doc.AddProperty("ModificationType", "mtTracePoint");
    doc.AddProperty("EventNumber", EventNumber);
    doc.AddProperty("TS", Sprintf("%.3f", TS));
    if (RouteUniqueID) {
        doc.AddProperty("RouteID", RouteUniqueID);
    }
    doc.AddProperty("RHASH", GetUniqueRouteHash());
    if (DID != 0) {
        doc.AddProperty("DID", DID);
    }
    for (const auto& kvPair : ExtProperties) {
        doc.AddProperty("__" + kvPair.first, kvPair.second);
    }
    Y_VERIFY(GeoShardingEnabled);
    if (GeoShardingEnabled) {
        doc.AddGeoData().AddCoordinate({Lat, Lon}).SetType("TRAFFIC_METRIC");
    }
    return action;
}

TString TMetrikaEvent::GetUniqueRouteHash() const {
    auto hash = CityHash128(UID.data(), UID.size());
    hash = CityHash128WithSeed(DID.data(), DID.size(), hash);
    hash = CityHash128WithSeed(RouteUniqueID.data(), RouteUniqueID.size(), hash);
    return HexEncode(&hash.first, sizeof(hash.first)) + "-"
         + HexEncode(&hash.second, sizeof(hash.second));
}

} // namespace NPq2Saas
