#include "egts.h"

#include <drive/pq2saas/liblogging/log.h>

#include <drive/contrib/cpp/telemetry/egts_service.hpp>

namespace {
    class TUUIDManager {
    public:
        using TServicePtr = TAtomicSharedPtr<telemetry::EGTSService>;

    public:
        static TServicePtr GetService(const TString& host, ui16 port, const TString& uuid) {
            return Singleton<TUUIDManager>()->GetServiceImpl(host, port, uuid);
        }

    private:
        TServicePtr GetServiceImpl(const TString& host, ui16 port, const TString& uuid) {
            const TString& key = host + ":" + ToString(port) + ":" + uuid;
            {
                TReadGuard guard(Lock);
                auto p = Services.find(key);
                if (p != Services.end()) {
                    return p->second;
                }
            }
            {
                TWriteGuard guard(Lock);
                auto p = Services.find(key);
                if (p != Services.end()) {
                    return p->second;
                }

                const std::string host_(host.begin(), host.end());
                const std::string imei_(uuid.begin(), uuid.end());
                const std::chrono::milliseconds timeout_(10000);
                auto service = MakeAtomicShared<telemetry::EGTSService>(host_, port, imei_, timeout_);
                Services.emplace(key, service);
                return service;
            }
        }

    private:
        TMap<TString, TServicePtr> Services;
        TRWMutex Lock;
    };
}

NPq2Saas::TEGTSAction::TEGTSAction(const TString& uuid, const TString& destinationName, TDependencyManagerPtr dependencyManager, NPq2SaasMonitoring::TManager::TDeliveryStatsPtr deliveryStats)
    : UUID(uuid)
    , DestinationName(destinationName)
    , DeliveryStats(deliveryStats)
    , DeliveryDestinationStats(deliveryStats->Stats.at(destinationName))
{
    const auto& destination = dependencyManager->GetResource<TGenericDestination>(DestinationName);
    Host = destination.GetHost();
    Port = destination.GetPort();
}

void NPq2Saas::TEGTSAction::Send(const TState& state) {
    auto& stats = *DeliveryDestinationStats;

    stats.EventsSentTotal->Inc();
    auto requestStartTime = Now();
    try {
        telemetry::CarState carState;
        carState.id = 0;
        carState.coords.lat = state.Latitude;
        carState.coords.lon = state.Longitude;
        carState.direction = state.Direction;
        carState.speed = state.Speed;
        carState.free = state.Free;
        carState.moving = state.Moving;
        carState.asn = state.Asn;
        carState.asnBusy = state.AsnBusy;
        carState.asnService = state.AsnService;
        carState.timestamp = std::chrono::system_clock::from_time_t(state.Timestamp.TimeT());

        auto service = TUUIDManager::GetService(Host, Port, UUID);
        Y_ENSURE(service);
        auto sendStartTime = Now();
        service->Send({ carState });
        auto finishTime = Now();

        stats.ReportSaasSendResult(HTTP_OK);
        stats.SaasReqProcessingTimeCounters.ReportEvent(requestStartTime);
        stats.EventsSentSucceded->Inc();
        LOG_JSON
            .KV("imei", UUID)
            .KV("success", "1")
            .KV("send_duration", (finishTime - sendStartTime).MicroSeconds())
            .KV("total_duration", (finishTime - requestStartTime).MicroSeconds())
            .KV("lat", state.Latitude)
            .KV("lon", state.Longitude)
            .KV("dir", state.Direction)
            .KV("speed", state.Speed)
            .KV("free", state.Free)
            .KV("moving", state.Moving)
            .KV("asn", state.Asn)
            .KV("asn_busy", state.AsnBusy)
            .KV("asn_service", state.AsnService)
            .KV("timestamp", state.Timestamp.Seconds())
        ;
    } catch (...) {
        stats.ReportSaasSendResult(HTTP_INTERNAL_SERVER_ERROR);
        stats.SaasFailedReqProcessingTimeCounters.ReportEvent(requestStartTime);
        stats.EventsSentFailedMaxRetries->Inc();
        LOG_JSON
            .KV("imei", UUID)
            .KV("success", "0")
            .KV("total_duration", (Now() - requestStartTime).MicroSeconds())
            .KV("lat", state.Latitude)
            .KV("lon", state.Longitude)
            .KV("dir", state.Direction)
            .KV("speed", state.Speed)
            .KV("free", state.Free)
            .KV("moving", state.Moving)
            .KV("asn", state.Asn)
            .KV("asn_busy", state.AsnBusy)
            .KV("asn_service", state.AsnService)
            .KV("timestamp", state.Timestamp.Seconds())
        ;
        throw;
    }
}
