#include "saas_indexing.h"

#include <drive/pq2saas/liblogging/log.h>
#include <drive/pq2saas/libprobes/actions.h>

namespace {
bool IsRetriableCode(ui16 code) {
    return code >= 500 && code < 600;
}
} // anonymous namespace

namespace NPq2Saas {

LWTRACE_USING(PQ2SAAS_ACTIONS_PROVIDER);

void TSaasIndexingAction::Process(void* /*threadSpecificResource*/) {
    Process();
}

void TSaasIndexingAction::Process() {
    size_t retry = 0;
    auto& stats = *DeliveryDestinationStats;
    NRTLine::TSendParams params = NRTLine::TSendParams().SetInstantReply(InstantReply);
    NRTLine::TSendResult sendResult;
    for (; retry < MaxRetries; ++retry) {
        stats.EventsSentTotal->Inc();
        auto startSendingToSaas = Now();
        sendResult = IndexingClient->SendJson(SaasAction, params);
        auto resultCode = sendResult.GetHttpCode();
        stats.ReportSaasSendResult(resultCode);
        auto resultMessage = sendResult.GetMessage();
        auto jsonMessage = SaasAction.BuildJsonMessage();

        LWPROBE(SaasIndexingResponseResult,
                DestinationName, (Now() - startSendingToSaas).MilliSeconds(),
                resultCode, UUID, jsonMessage, resultMessage);

        TToJsonContext jsonContext(TToJsonContext::COMMON_JSON);
        LOG_JSON.KV("method", "TSaasIndexingAction::Process")
                .KV("json_message", SaasAction.ToJson(jsonContext))
                .KV("result_code", resultCode)
                .KV("result_message", resultMessage);

        if (sendResult.GetCode() == NRTLine::TSendResult::srOK) {
            stats.SaasReqProcessingTimeCounters.ReportEvent(startSendingToSaas);
            stats.EventLatencyCounters.ReportEvent(TInstant::Seconds(PQEventTime));
            stats.EventPreSaasLifetimeCounters.ReportEvent(TInstant::Seconds(RealEventTime));
            stats.EventsSentSucceded->Inc();
            break;
        }
        stats.SaasFailedReqProcessingTimeCounters.ReportEvent(startSendingToSaas);
        if (!IsRetriableCode(resultCode)) {
            stats.EventsSentFailedNonRetriable->Inc();
            break;
        }
        if (retry + 1 == MaxRetries) {
            stats.EventsSentFailedMaxRetries->Inc();
        } else {
            stats.EventsSentFailedAndRetried->Inc();
        }
    }
    stats.ReqPerEventCounters[retry]->Inc();
    SendResult = sendResult;
}

} // namespace NPq2Saas
