#include "analyzer_location.h"

#include <drive/pq2saas/libevent/event.h>
#include <drive/pq2saas/libhandling/actions/saas_indexing.h>
#include <drive/pq2saas/libprobes/handlers.h>

namespace NPq2Saas {

LWTRACE_USING(PQ2SAAS_HANDLERS_ANALYZER_LOCATION_PROVIDER);

void TAnalyzerLocationEventHandler::OnEvent(const THashMap<TString, TString>& item) {
    for (const auto& field : {"uuid", "clienttime", "receive_time", "lat", "lon"}) {
        if (!item.contains(field)) {
            throw TBadInputException() << "No field: " << field;
        }
    }
    const auto& settings = HandlerSettings.Get<TAnalyzerLocationHandlerSettings>();
    const auto& destinationName = settings.SaasIndexingDestinationName;
    const auto& deliveryStats = MonManager.GetDeliveryStats(DeliveryName);
    const auto& uuid = item.at("uuid");
    auto saasEvent = MakeAtomicShared<TLocationEvent>(uuid);
    auto eventTimeStr = item.at("clienttime");
    auto receiveTimeStr = item.at("receive_time");
    ui64 pqEventTime = 0;
    ui64 receiveEventTime = 0;
    double lat = 0.0;
    double lon = 0.0;
    if (uuid.empty() ||
        !TryFromString(item.at("lat"), lat) || !TryFromString(item.at("lon"), lon) ||
        !TryFromString(eventTimeStr, pqEventTime) ||
        !TryFromString(receiveTimeStr, receiveEventTime))
    {
        MonManager.LogFormatErrors->Inc();
        throw TBadInputException() << "Wrong log format";
    }
    if (receiveEventTime < pqEventTime) {
        MonManager.EventsFromFuture->Inc();
        LWPROBE(EventsFromFuture, uuid, receiveEventTime, pqEventTime);
        throw TBadInputException() << "Events from future";
    }
    TDuration pqEventLag = Now() - TInstant::Seconds(pqEventTime);
    MonManager.EventInitialLag.ReportEvent(pqEventLag);
    if (pqEventLag >= settings.Filtering.EventInitialLagThreshold) {
        deliveryStats->TooOldEventsCount->Inc();
        return;
    }
    saasEvent->SetTS(eventTimeStr);
    saasEvent->SetPosition(lat, lon);
    if (!saasEvent->HasValidPosition()) {
        MonManager.EventsWithInvalidPosition->Inc();
        throw TBadInputException() << "Events with invalid position";
    }
    auto sender = MakeAtomicShared<TSaasIndexingAction>(
        pqEventTime, static_cast<ui64>(saasEvent->GetTS()), saasEvent->GetUID(),
        saasEvent->ToSaasAction(),
        destinationName, DependencyManager, deliveryStats
    );
    (void)Queue->AddFunc([sender]{ sender->Process(); });
}

} // namespace NPq2Saas
