#include "bs_mobile.h"

#include <drive/pq2saas/libevent/event.h>
#include <drive/pq2saas/libhandling/actions/saas_indexing.h>
#include <drive/pq2saas/libhandling/actions/saas_search.h>
#include <drive/pq2saas/libprobes/handlers.h>
#include <zootopia/library/pb/mobile_metrics/mobile_metrics_data.pb.h>

#include <library/cpp/json/json_reader.h>

#include <util/generic/hash_set.h>
#include <util/generic/yexception.h>
#include <util/string/builder.h>
#include <util/string/printf.h>

#include <cmath>


namespace {

using namespace NPq2Saas;

bool IsNeededEvent(const TFilteringSettings& settings,
                   const NMobileMetrics::ReportMessage::Session::Event& event)
{
    return event.type() == NMobileMetrics::ReportMessage_Session_Event_EventType_EVENT_CLIENT &&
           settings.IsNeededEventName(event.name());
}

double CorrectClientEventTime(double eventTime, ui64 sendTime, ui64 receiveTime) {
    // we assume that under normal conditions receiveTime == sendTime
    // however this can be not true if client device has incorrect system time
    // to correct device time error we calculate this error and apply to eventTime.
    //
    // @note: don't use parentheses, because (ui64 - ui64) is subject to underflow
    //
    return eventTime + receiveTime - sendTime;
}

struct TKvSearchResult {
    bool TrackingAllowed;
    bool UuidFound;
};

bool CanProceedWithAccountId(const TString& value) {
    NJson::TJsonValue jsonData;
    TStringInput strInput(value);
    if (!NJson::ReadJsonTree(&strInput, &jsonData)) {
        return false;
    }
    if (jsonData.Has("status") && jsonData["status"].GetStringRobust() == "allowed") {
        return true;
    }
    return false;
}

TKvSearchResult ParseSearchReply(const NRTLine::TSearchReply& reply) {
    TKvSearchResult result{false, false};

    if (!reply.IsSucceeded()) {
        return result;
    }

    for (const auto& groupping : reply.GetReport().GetGrouping()) {
        for (const auto& group : groupping.GetGroup()) {
            for (const auto& doc : group.GetDocument()) {
                if (doc.GetUrl().EndsWith("-ts")) {
                    // check if tracking is allowed
                    for (const auto& kvPair : doc.GetArchiveInfo().GetGtaRelatedAttribute()) {
                        if (kvPair.GetKey() == "value") {
                            result.TrackingAllowed = CanProceedWithAccountId(kvPair.GetValue());
                        }
                    }
                } else if (doc.GetUrl().StartsWith("uuid_")) {
                    result.UuidFound = true;
                } else {
                    ythrow yexception() << "Unexpected document url: " << doc.GetUrl();
                }
            }
        }
    }
    return result;
}

const TString SAAS_SEARCH_EXTRA = "&timeout=30000";
const TDuration SAAS_TIMEOUT = TDuration::MilliSeconds(30);

} // anonymous namespace

namespace NPq2Saas {

LWTRACE_USING(PQ2SAAS_HANDLERS_BS_MOBILE_PROVIDER);

void TBsMobileEventHandler::OnEvent(const THashMap<TString, TString>& item) {
    for (const auto& field : {"eventtime", "data"}) {
        if (!item.contains(field)) {
            throw TBadInputException() << "No field: " << field;
        }
    }
    ui64 pqEventTime = FromString(item.at("eventtime"));
    TStringBuf dataValue = item.at("data");
    NMobileMetrics::ReportMessage msg;
    if (!msg.ParseFromArray(dataValue.data(), dataValue.length())) {
        MonManager.ProtoParsingErrors->Inc();
        throw yexception() << "Proto parsing error";
    }
    auto appId = msg.report_request_parameters().app_id();
    if (HandlerSettings.Get<TBsMobileHandlerSettings>().Filtering.IsNeededAppId(appId)) {
        MonManager.MatchingProtos->Inc();
        SendTracksInfo(pqEventTime, msg);
    } else {
        LWPROBE(NonMatchingProto, pqEventTime, msg.ShortUtf8DebugString());
        MonManager.NonMatchingProtos->Inc();
    }
}

void TBsMobileEventHandler::SendTracksInfo(ui64 pqEventTime,
                                           const NMobileMetrics::ReportMessage& msg)
{
    const auto& reqParams = msg.report_request_parameters();
    const auto& uuid = reqParams.uuid();
    const auto& settings = HandlerSettings.Get<TBsMobileHandlerSettings>();
    const auto& deliveryStats = MonManager.GetDeliveryStats(DeliveryName);
    if (settings.Spy.ShouldSpyOnUUID(uuid)) {
        TStringBuilder out;
        msg.PrintJSON(out.Out);
        Cout << out << Endl;
    }
    for (const auto& session : msg.sessions()) {
        for (const auto& event : session.events()) {
            MonManager.EventsWithNeededAppId->Inc();
            if (!IsNeededEvent(settings.Filtering, event)) {
                LWPROBE(NotNeededEvent, uuid, pqEventTime, ToString(msg), ToString(event));
                continue;
            }
            auto saasEvent = MakeAtomicShared<TMetrikaEvent>(uuid, reqParams.device_id());
            if (event.has_account()) {
                saasEvent->SetAccountID(event.account().id());
                MonManager.EventsWithAccountId->Inc();
            } else {
                MonManager.EventsWithoutAccountId->Inc();
                LWPROBE(NoAccountId, uuid, pqEventTime, ToString(msg), ToString(event));
                continue;
            }
            MonManager.EventsOfNeededType->Inc();
            saasEvent->SetType(event.name());
            saasEvent->SetEventNumber(event.number());
            MonManager.EventName2Counter[event.name()]->Inc();
            NJson::TJsonValue clientData;
            TStringInput strInput(event.value());
            if (!NJson::ReadJsonTree(&strInput, &clientData)) {
                MonManager.JsonParsingErrors->Inc();
                LWPROBE(JsonParsingFailed, uuid, pqEventTime, ToString(msg), ToString(event));
                continue;
            }
            try {
                auto correctEventTS = CorrectClientEventTime(
                    FromString(clientData["timestamp"].GetString()),
                    msg.send_time().timestamp(), msg.receive_time().timestamp()
                );
                saasEvent->SetTS(Sprintf("%.3f", correctEventTS));
                saasEvent->SetPosition(42.0, 42.0); // mobile metrica coordinates are skipped while indexing
                saasEvent->SetRouteUniqueID(clientData["route_id"].GetString());
            } catch (const std::exception& e) {
                TStringBuilder out;
                out << FormatExc(e) << "\nEvent proto:\n" << event;
                Cerr << out << Endl;
                MonManager.JsonDocumentFormatErrors->Inc();
                LWPROBE(JsonDocumentFormatError, uuid, pqEventTime, ToString(msg), ToString(event));
                continue;
            }
            TInstant saasEventTS = TInstant::Seconds(saasEvent->GetTS());
            if ((Now() - saasEventTS) >= settings.Filtering.EventInitialLagThreshold) {
                deliveryStats->TooOldEventsCount->Inc();
                LWPROBE(TooOldEvent, uuid, pqEventTime, ToString(msg), ToString(event), saasEventTS.Seconds());
                continue;
            }
            saasEvent->SetExtProperties(clientData);

            deliveryStats->MetrikaEventLogBrokerTime.ReportEvent(TInstant::Seconds(msg.receive_time().timestamp()));
            deliveryStats->MetrikaEventSendTime.ReportEvent(TInstant::Seconds(msg.send_time().timestamp()));

            const auto& indexDestinationName = settings.SaasIndexingDestinationName;
            const auto& searchUserKVName = settings.SaasSearchUserKVName;
            const auto& indexUserKVName = settings.SaasIndexingUserKVName;

            if (settings.GeoShardingEnabled) {
                saasEvent->EnableGeoSharding();
            }

            auto checker = MakeAtomicShared<TSaasSearchAction>(
                saasEvent->GetUID(),
                NRTLine::TQuery().SetText(event.account().id() + "-ts,uuid_" + saasEvent->GetUID())
                               .SetTimeout(SAAS_TIMEOUT)
                               .SetExtraParams(SAAS_SEARCH_EXTRA),
                searchUserKVName, DependencyManager, deliveryStats
            );

            auto sender = MakeAtomicShared<TSaasIndexingAction>(
                pqEventTime, saasEventTS.Seconds(), saasEvent->GetUID(),
                saasEvent->ToSaasAction(),
                indexDestinationName, DependencyManager, deliveryStats
            );

            NRTLine::TAction userKvSenderAction;
            userKvSenderAction.GetDocument().AddSpecialKey("accid", event.account().id())
                                            .AddProperty("accid", event.account().id())
                                            .SetUrl("uuid_" + saasEvent->GetUID());
            auto userKvSender = MakeAtomicShared<TSaasIndexingAction>(
                pqEventTime, saasEventTS.Seconds(), saasEvent->GetUID(), std::move(userKvSenderAction),
                indexUserKVName, DependencyManager, deliveryStats
            );

            (void)Queue->AddFunc([checker, sender, userKvSender, deliveryStats, saasEventTS]{
                // search for acc_id and uuid
                auto checkResult = ParseSearchReply(checker->Process());

                if (checkResult.TrackingAllowed) {
                    // send MobileMetrika event to SaaS
                    deliveryStats->SaasEventTs.ReportEvent(saasEventTS);
                    sender->Process();
                    // send UUID if it hasn't been indexed yet
                    if (!checkResult.UuidFound) {
                        userKvSender->Process();
                    }
                } else {
                    deliveryStats->EventsWithForbiddenAccountsCount->Inc();
                    // delete UUID if it exists
                    if (checkResult.UuidFound) {
                        userKvSender->MutateSaasAction().SetActionType(NRTLine::TAction::atDelete);
                        userKvSender->Process();
                    }
                }
            });
        }
    }
}

} // namespace NPq2Saas
