#include "bs_mobile_feedback.h"

#include <drive/pq2saas/libprobes/handlers.h>

#include <zootopia/library/pb/mobile_metrics/mobile_metrics_data.pb.h>

#include <yweb/news/fetcher_lib/fetcher.h>

#include <library/cpp/json/json_reader.h>
#include <util/generic/hash_set.h>
#include <util/string/builder.h>

#include <cmath>

namespace {

using namespace NPq2Saas;

bool IsNeededEvent(const TFilteringSettings& settings,
                   const NMobileMetrics::ReportMessage::Session::Event& event)
{
    return event.type() == NMobileMetrics::ReportMessage_Session_Event_EventType_EVENT_CLIENT &&
           settings.IsNeededEventName(event.name());
}

} // anonymous namespace

namespace NPq2Saas {

LWTRACE_USING(PQ2SAAS_HANDLERS_BS_MOBILE_FEEDBACK_PROVIDER);

void TBsMobileEventFeedbackHandler::OnEvent(const THashMap<TString, TString>& item) {
    for (const auto& field : {"eventtime", "data"}) {
        if (!item.contains(field)) {
            throw TBadInputException() << "No field: " << field;
        }
    }
    ui64 pqEventTime = FromString(item.at("eventtime"));
    TStringBuf dataValue = item.at("data");
    NMobileMetrics::ReportMessage msg;
    if (!msg.ParseFromArray(dataValue.data(), dataValue.length())) {
        MonManager.ProtoParsingErrors->Inc();
    }
    auto appId = msg.report_request_parameters().app_id();
    if (HandlerSettings.Get<TBsMobileFeedbackHandlerSettings>().Filtering.IsNeededAppId(appId)) {
        MonManager.MatchingProtos->Inc();
        SendTracksInfo(pqEventTime, msg);
    } else {
        MonManager.NonMatchingProtos->Inc();
    }
}

void TBsMobileEventFeedbackHandler::SendTracksInfo(ui64 /* pqEventTime */,
                                                   const NMobileMetrics::ReportMessage& msg)
{
    const auto& reqParams = msg.report_request_parameters();
    const auto& settings = HandlerSettings.Get<TBsMobileFeedbackHandlerSettings>();
    const auto& deliveryStats = MonManager.GetDeliveryStats(DeliveryName);
    const auto& destinationStats = deliveryStats->Stats.at(settings.HttpRequestDestinationName);
    if (settings.Spy.ShouldSpyOnUUID(reqParams.uuid())) {
        TStringBuilder out;
        msg.PrintJSON(out.Out);
        Cout << out << Endl;
    }
    for (const auto& session : msg.sessions()) {
        for (const auto& event : session.events()) {
            MonManager.EventsWithNeededAppId->Inc();
            if (!IsNeededEvent(settings.Filtering, event)) {
                continue;
            }
            (void)Queue->AddFunc([&settings, event, destinationStats]{
                ui64 retry = 0;
                for (const auto& url : settings.SaasFeedbackBackends) {
                    ui16 httpcode = 0;
                    try {
                        NHttpFetcher::TRequestRef request = new NHttpFetcher::TRequest(url);
                        request->PostData = event.value();
                        request->Deadline = TInstant::Now() + TDuration::Seconds(2);
                        auto result = NHttpFetcher::FetchNow(request);
                        httpcode = result->Code;
                        LWPROBE(BackendResponse, url, httpcode, event.value(), result->Data);
                    } catch (const std::exception& e) {
                        LWPROBE(BackendResponse, url, 0, event.value(), FormatExc(e));
                    }

                    destinationStats->ReportSaasSendResult(httpcode);
                    if (httpcode == 200) {
                        destinationStats->EventsSentSucceded->Inc();
                        break;
                    } else {
                        if (retry + 1 == settings.SaasFeedbackBackends.size()) {
                            destinationStats->EventsSentFailedMaxRetries->Inc();
                        } else {
                            destinationStats->EventsSentFailedAndRetried->Inc();
                        }
                    }
                    ++retry;
                }
                Y_ASSERT(retry < destinationStats->ReqPerEventCounters.size());
                destinationStats->ReqPerEventCounters[retry]->Inc();
            });
        }
    }
}

} // namespace NPq2Saas
