#include "abstract.h"

namespace NPq2Saas {

    TAtomic TOneFieldRemapItem::FieldNotExistsCounter = 0;

    bool TComplexUrlRemapItem::DoRemap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const {
        auto itF = item.find(PQPropertyNameFirst);
        auto itS = item.find(PQPropertyNameSecond);

        if (itS == item.end() || itF == item.end()) {
            if (!IsNecessary) {
                return true;
            }
            ythrow yexception() << "Can't find properties for complex url: " << PQPropertyNameFirst << "/" << PQPropertyNameSecond;
        }
        doc.SetUrl(itF->second + "_" + itS->second);
        return true;
    }

    bool TGeoBaseRemapItem::DoRemap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const {
        auto itX = item.find(PQPropertyNameX);
        auto itY = item.find(PQPropertyNameY);

        if (itX == item.end() || itY == item.end()) {
            if (!IsNecessary) {
                return true;
            }
            ythrow yexception() << "Can't find properties for geo info: " << PQPropertyNameX << "/" << PQPropertyNameY;
        }
        TGeoCoord c;
        if (!TryFromString<double>(itX->second, c.X) || !TryFromString<double>(itY->second, c.Y)) {
            ythrow yexception() << "Can't convert properties to <double> for geo info: " << PQPropertyNameX << "/" << PQPropertyNameY;
        }
        auto& geoData = doc.AddGeoData().AddCoordinate(c);
        if (!!Type) {
            geoData.SetType(Type);
        }
        return true;
    }

    bool TDeadlineRemapItem::DoRemapOneField(const TString& value, NRTLine::TDocument& doc) const {
        ui32 ts;
        if (!TryFromString<ui32>(value, ts)) {
            ythrow yexception() << "Incorrect type for property " << PQPropertyName << ": " << value;
        }
        doc.SetDeadline(TInstant::Seconds(ts));
        return true;
    }

    bool TTimestampRemapItem::DoRemapOneField(const TString& value, NRTLine::TDocument& doc) const {
        ui32 ts;
        if (!TryFromString<ui32>(value, ts)) {
            ythrow yexception() << "Incorrect ts format: " << value;
        }
        doc.SetTimestamp(ts);
        return true;
    }

    bool TOneFieldRemapItem::DoRemap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const {
        auto it = item.find(PQPropertyName);
        if (it == item.end()) {
            if (IsNecessary) {
                AtomicIncrement(FieldNotExistsCounter);
                ythrow yexception() << "Can't find necessary field " << PQPropertyName;
            }
            return true;
        }
        return DoRemapOneField(it->second, doc);
    }

    bool IRTLineConverter::Convert(const THashMap<TString, TString>& item, NRTLine::TAction& saasAction) const {
        saasAction.SetActionType(NRTLine::TAction::atModify);
        NRTLine::TDocument& doc = saasAction.AddDocument();
        for (auto&& i : GetRemapInfo()) {
            if (!i->Remap(item, doc)) {
                return false;
            }
        }
        return true;
    }

    bool IRemapItem::Remap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const {
        try {
            return DoRemap(item, doc);
        } catch (const std::exception& e) {
            ERROR_LOG << "Parsing exception: " << FormatExc(e) << Endl;
            return false;
        }
    }

    bool TPatternUrlRemapItem::DoRemap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const {
        TString url = Pattern;
        for (const auto& name : PQPropertyNames) {
            auto it = item.find(name);
            if (it == item.end()) {
                if (!IsNecessary) {
                    return true;
                }
                ythrow yexception() << "Can't find property " << name << " for template url";
            }
            size_t pos = url.find('$');
            CHECK_WITH_LOG(pos < url.size());
            url.replace(pos, 1, it->second);
        }

        doc.SetUrl(url);
        return true;
    }
}
