#pragma once
#include <util/generic/hash.h>
#include <util/generic/string.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <rtline/api/document.h>
#include <rtline/api/action.h>

namespace NPq2Saas {

    class IRemapItem {
    protected:
        virtual bool DoRemap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const = 0;
    public:

        virtual ~IRemapItem() { }

        using TPtr = TAtomicSharedPtr<IRemapItem>;

        virtual bool Remap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const;
    };

    class IRTLineConverter {
    protected:
        virtual const TVector<IRemapItem::TPtr>& GetRemapInfo() const = 0;
    public:

        virtual ~IRTLineConverter() {

        }

        bool Convert(const THashMap<TString, TString>& item, NRTLine::TAction& saasAction) const;
    };

    class TBaseRemapItem: public IRemapItem {
    protected:
        bool IsNecessary = true;
    public:

        TBaseRemapItem& SetIsNecessary(const bool value) {
            IsNecessary = value;
            return *this;
        }
    };

    class TOneFieldRemapItem: public TBaseRemapItem {
    private:
        static TAtomic FieldNotExistsCounter;
    protected:
        const TString PQPropertyName;
        virtual bool DoRemapOneField(const TString& value, NRTLine::TDocument& doc) const = 0;
    public:

        static i64 GetFieldNotExistsCounter() {
            return AtomicGet(FieldNotExistsCounter);
        }

        TOneFieldRemapItem(const TString& pqPropertyName)
            : PQPropertyName(pqPropertyName) {
            CHECK_WITH_LOG(!!pqPropertyName);
        }

        virtual bool DoRemap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const override;
    };

    class TUrlRemapItem: public TOneFieldRemapItem {
    protected:
        virtual bool DoRemapOneField(const TString& value, NRTLine::TDocument& doc) const override {
            doc.SetUrl(value);
            return true;
        }
    public:
        TUrlRemapItem(const TString& pqPropertyName)
            : TOneFieldRemapItem(pqPropertyName) {
        }
    };

    class TTimestampRemapItem: public TOneFieldRemapItem {
    protected:
        virtual bool DoRemapOneField(const TString& value, NRTLine::TDocument& doc) const override;
    public:
        TTimestampRemapItem(const TString& pqPropertyName)
            : TOneFieldRemapItem(pqPropertyName) {
        }
    };

    class TDeadlineRemapItem: public TOneFieldRemapItem {
    protected:
        virtual bool DoRemapOneField(const TString& value, NRTLine::TDocument& doc) const override;
    public:
        TDeadlineRemapItem(const TString& pqPropertyName)
            : TOneFieldRemapItem(pqPropertyName) {
        }
    };

    class TDefaultLiveTimeSetter: public IRemapItem {
    private:
        const TDuration DefaultLiveTime;
    protected:

        virtual bool DoRemap(const THashMap<TString, TString>& /*item*/, NRTLine::TDocument& doc) const override {
            doc.SetDeadline(Now() + DefaultLiveTime);
            return true;
        }
    public:

        TDefaultLiveTimeSetter(const TDuration defaultLiveTime)
            : DefaultLiveTime(defaultLiveTime)
        {

        }
    };

    class TAdditionalProperty: public IRemapItem {
    private:
        const TString PropName;
        const TString PropValue;
    protected:

        virtual bool DoRemap(const THashMap<TString, TString>& /*item*/, NRTLine::TDocument& doc) const override {
            doc.AddProperty(PropName, PropValue);
            return true;
        }
    public:
        TAdditionalProperty(const TString& propName, const TString& propValue)
            : PropName(propName)
            , PropValue(propValue) {

        }

    };

    template <class TValidation>
    class TPropertyRemapItem: public TOneFieldRemapItem {
    protected:
        TString ExportPropertyName;
        virtual bool DoRemapOneField(const TString& value, NRTLine::TDocument& doc) const override {
            TValidation checkValue;
            if (!TryFromString<TValidation>(value, checkValue)) {
                ythrow yexception() << "Incorrect type for property " << PQPropertyName << ": " << value;
            }
            doc.AddProperty(ExportPropertyName, value);
            return true;
        }
    public:

        TPropertyRemapItem(const TString& pqPropertyName)
            : TOneFieldRemapItem(pqPropertyName)
            , ExportPropertyName(pqPropertyName) {

        }

        TPropertyRemapItem(const TString& pqPropertyName, const TString& exportPropertyName)
            : TOneFieldRemapItem(pqPropertyName)
            , ExportPropertyName(exportPropertyName) {
            CHECK_WITH_LOG(!!ExportPropertyName);
        }
    };

    template <>
    class TPropertyRemapItem<TString>: public TOneFieldRemapItem {
    protected:
        TString PQPropertyName;
        TString ExportPropertyName;
        virtual bool DoRemapOneField(const TString& value, NRTLine::TDocument& doc) const override {
            doc.AddProperty(ExportPropertyName, value);
            return true;
        }
    public:
        TPropertyRemapItem(const TString& pqPropertyName)
            : TOneFieldRemapItem(pqPropertyName)
            , ExportPropertyName(pqPropertyName) {

        }

        TPropertyRemapItem(const TString& pqPropertyName, const TString& exportPropertyName)
            : TOneFieldRemapItem(pqPropertyName)
            , ExportPropertyName(exportPropertyName) {
            CHECK_WITH_LOG(!!ExportPropertyName);
        }
    };

    class TGeoBaseRemapItem: public TBaseRemapItem {
    protected:
        TString PQPropertyNameX;
        TString PQPropertyNameY;
        const TString Type;
        virtual bool DoRemap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const override;
    public:
        TGeoBaseRemapItem(const TString& pqPropertyNameX, const TString& pqPropertyNameY, const TString& type)
            : PQPropertyNameX(pqPropertyNameX)
            , PQPropertyNameY(pqPropertyNameY)
            , Type(type)
        {
            CHECK_WITH_LOG(!!pqPropertyNameX);
            CHECK_WITH_LOG(!!pqPropertyNameY);
        }
    };

    class TComplexUrlRemapItem: public TBaseRemapItem {
    protected:
        TString PQPropertyNameFirst;
        TString PQPropertyNameSecond;
        virtual bool DoRemap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const override;
    public:
        TComplexUrlRemapItem(const TString& pqPropertyNameFirst, const TString& PQPropertyNameSecond)
            : PQPropertyNameFirst(pqPropertyNameFirst)
            , PQPropertyNameSecond(PQPropertyNameSecond)
        {
            CHECK_WITH_LOG(!!pqPropertyNameFirst);
            CHECK_WITH_LOG(!!PQPropertyNameSecond);
        }
    };

    class TPatternUrlRemapItem : public TBaseRemapItem {
    protected:
        TVector<TString> PQPropertyNames;
        TString Pattern;
        virtual bool DoRemap(const THashMap<TString, TString>& item, NRTLine::TDocument& doc) const override;
    public:
        TPatternUrlRemapItem(const TString& pattern, const TVector<TString>& pqPropertyNames)
            : PQPropertyNames(pqPropertyNames)
            , Pattern(pattern)
        {
            for (const auto& name : pqPropertyNames) {
                CHECK_WITH_LOG(!!name);
            }
        }
    };
} // namespace NPq2Saas
