#include "backend_events_handler.h"
#include "backend_events_settings.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/unistat/unistat.h>
#include <util/string/join.h>
#include <drive/pq2saas/libhandling/actions/saas_indexing.h>

namespace {
    using TActionPtr = THolder<NRTLine::TAction>;

    TActionPtr MakeAction(const TString& url, const TString& userId, ui32 timestamp,
                          const TDuration& ttl) {
        TActionPtr action = MakeHolder<NRTLine::TAction>();
        NRTLine::TDocument& doc = action->AddDocument();
        doc.SetUrl(url);
        doc.SetTimestamp(timestamp);
        doc.AddProperty("Timestamp", timestamp);
        doc.AddSpecialKey("UserId", userId);
        if (ttl != TDuration::Zero())
            doc.SetDeadline(Now() + ttl);
        return action;
    }

    TActionPtr MakeLocationAction(const TString& userId,
                                  const THashMap<TString, TString>& /*item*/,
                                  const NJson::TJsonValue& jsonData,
                                  const NPq2Saas::TDriveBackendEventsHandlerSettings& settings,
                                  const TString& requestId, ui32 timestamp) {
        auto latValue = jsonData.GetValueByPath("headers.Lat");
        if (!latValue) {
            latValue = jsonData.GetValueByPath("headers.lat");
        }
        auto lonValue = jsonData.GetValueByPath("headers.Lon");
        if (!lonValue) {
            lonValue = jsonData.GetValueByPath("headers.lon");
        }
        if (latValue && lonValue) {
            double lat = FromString(latValue->GetString());
            double lon = FromString(lonValue->GetString());
            TActionPtr action = MakeAction("user_id:" + userId + ":location", userId,
                                           timestamp, settings.GetTTL());
            auto& doc = action->GetDocument();
            doc.AddProperty("Latitude", lat);
            doc.AddProperty("Longitude", lon);
            doc.AddGeoData().AddCoordinate({lat, lon});
            return action;
        } else {
            DEBUG_LOG << "Missing location data for request " << requestId << Endl;
        }
        return nullptr;
    }

    TActionPtr MakeDataAction(const TString& userId,
                              const THashMap<TString, TString>& item,
                              const NJson::TJsonValue& jsonData,
                              const NPq2Saas::TDriveBackendEventsHandlerSettings& settings,
                              const TString& requestId, ui32 timestamp) {
        TActionPtr action = MakeAction("user_id:" + userId + ":data", userId,
                                       timestamp, settings.GetTTL());
        auto& doc = action->GetDocument();
        AddItemProperties(settings.GetExportFields(), item, doc, requestId);
        AddJsonProperties(settings.GetExportDataFields(), item.at("source"), jsonData, doc,
                          requestId);
        NPq2Saas::AddExperiments(jsonData, doc, requestId);
        return action;
    }
}  // namespace

namespace NPq2Saas {
    void TDriveBackedEventHandler::OnEvent(const THashMap<TString, TString>& item) {
        const auto& settings = HandlerSettings.Get<TDriveBackendEventsHandlerSettings>();
        if (item.at("event") != "Authorization") {
            return;
        }

        if (settings.GetSourceRegExp().GetRegExpr() &&
            !settings.GetSourceRegExp().Match(item.at("source").data())) {
            return;
        }

        const TString& userId = item.at("user_id");
        const ui32 timestamp = FromString(item.at("unixtime"));
        const TString& requestId = item.at("reqid");
        const auto data = NJson::ReadJsonFastTree(item.at("data"));
        const auto& deliveryStats = MonManager.GetDeliveryStats(DeliveryName);

        for (auto&& f : {MakeLocationAction, MakeDataAction}) {
            TActionPtr action = f(userId, item, data, settings, requestId, timestamp);
            if (action) {
                auto sender = MakeHolder<TSaasIndexingAction>(
                        Now().Seconds(), timestamp, action->GetDocument().GetUrl(),
                        std::move(*action), settings.GetDestinationName(), DependencyManager,
                        deliveryStats);
                Queue->SafeAddAndOwn(THolder(sender.Release()));
                TUnistat::Instance().PushSignalUnsafe(BackendEventsSendSignalName(DeliveryName), 1);
            }
        }
    }

    IEventHandler::TFactory::TRegistrator<TDriveBackedEventHandler> DriveBackendEvents(
            NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_BACKEND_EVENTS);
}  // namespace NPq2Saas
