#include "backend_events_history_handler.h"
#include "backend_events_settings.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/unistat/unistat.h>
#include <drive/library/cpp/user_events_api/user_requests_history.h>
#include <drive/pq2saas/libhandling/actions/saas_indexing.h>

namespace {
    using TActionPtr = THolder<NRTLine::TAction>;
    using TUserRequestData = NDrive::TUserRequestData;

    struct TBaseDocumentFields {
        TString UserId;
        ui32 Timestamp;
        TString EventType;
        TString RequestId;
        TDuration TTL;
    };

    TActionPtr MakeAction(const TBaseDocumentFields& fields) {
        TActionPtr action = MakeHolder<NRTLine::TAction>();
        NRTLine::TDocument& doc = action->AddDocument();
        const TString url = TStringBuilder() << fields.EventType << ":" << fields.RequestId;
        doc.SetUrl(url);
        doc.SetTimestamp(fields.Timestamp);
        doc.AddSearchAttribute("timeround", (fields.Timestamp / 1000) * 1000);
        doc.AddSearchAttribute("Timestamp", fields.Timestamp);
        doc.AddGroupAttribute("Timestamp", fields.Timestamp);
        doc.AddSearchAttribute("UserId", fields.UserId);
        if (fields.TTL != TDuration::Zero()) {
            doc.SetDeadline(Now() + fields.TTL);
        }
        return action;
    }

    TActionPtr MakeAuthAction(const TBaseDocumentFields& baseFields,
                              const THashMap<TString, TString>& item,
                              const NPq2Saas::TDriveBackendEventsHistoryHandlerSettings& settings) {
        TActionPtr action = MakeAction(baseFields);
        auto& doc = action->GetDocument();
        AddItemProperties(settings.GetAuthExportFields(), item, doc, baseFields.RequestId);

        const auto jsonData = NJson::ReadJsonFastTree(item.at("data"));
        AddJsonProperties(settings.GetAuthExportDataFields(), item.at("source"), jsonData, doc,
                          baseFields.RequestId);

        auto latValue = jsonData.GetValueByPath("headers.Lat");
        if (!latValue) {
            latValue = jsonData.GetValueByPath("headers.lat");
        }
        auto lonValue = jsonData.GetValueByPath("headers.Lon");
        if (!lonValue) {
            lonValue = jsonData.GetValueByPath("headers.lon");
        }
        if (latValue && lonValue) {
            double lat = FromString(latValue->GetString());
            double lon = FromString(lonValue->GetString());
            doc.AddProperty("Latitude", lat);
            doc.AddProperty("Longitude", lon);
        }

        NPq2Saas::AddExperiments(jsonData, doc, baseFields.RequestId);
        return action;
    }

    TActionPtr MakeResponseAction(const TBaseDocumentFields& baseFields,
                                  const THashMap<TString, TString>& item,
                                  const NPq2Saas::TDriveBackendEventsHistoryHandlerSettings& settings) {
        TActionPtr action = MakeAction(baseFields);
        auto& doc = action->GetDocument();
        ui16 code = FromString(item.at("code"));
        doc.AddProperty("Code", code);
        if (code != 200 && item.find("data") != item.end()) {
            doc.AddProperty("Data", item.at("data"));
        }
        AddItemProperties(settings.GetResponseExportFields(), item, doc, baseFields.RequestId);
        return action;
    }
}  // namespace

namespace NPq2Saas {
    void TDriveBackedEventHistoryHandler::OnEvent(const THashMap<TString, TString>& item) {
        const auto& settings = HandlerSettings.Get<TDriveBackendEventsHistoryHandlerSettings>();
        const TString& event = item.at("event");
        const bool isAuthorization = TUserRequestData::IsAuthorizationEvent(event);
        if (!isAuthorization && !TUserRequestData::IsResponseEvent(event)) {
            return;
        }

        if (settings.GetSourceRegExp().GetRegExpr() &&
            !settings.GetSourceRegExp().Match(item.at("source").data())) {
            return;
        }

        TBaseDocumentFields baseFields;
        // Timeout events don't have user_id yet.
        if (item.find("user_id") == item.end()) {
            return;
        }

        baseFields.UserId = item.at("user_id");
        baseFields.Timestamp = FromString(item.at("unixtime"));
        baseFields.RequestId = item.at("reqid");
        baseFields.EventType = event;
        baseFields.TTL = settings.GetTTL();

        TActionPtr action;
        if (isAuthorization) {
            action = MakeAuthAction(baseFields, item, settings);
        } else {
            action = MakeResponseAction(baseFields, item, settings);
        }
        const auto& deliveryStats = MonManager.GetDeliveryStats(DeliveryName);
        auto sender = MakeHolder<TSaasIndexingAction>(
                Now().Seconds(), baseFields.Timestamp, action->GetDocument().GetUrl(),
                std::move(*action), settings.GetDestinationName(), DependencyManager,
                deliveryStats);
        Queue->SafeAddAndOwn(THolder(sender.Release()));
        TUnistat::Instance().PushSignalUnsafe(BackendEventsSendSignalName(DeliveryName), 1);
    }

    IEventHandler::TFactory::TRegistrator<TDriveBackedEventHistoryHandler> DriveBackendEventsHistory(
            NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_BACKEND_EVENTS_HISTORY);
}  // namespace NPq2Saas
