#include "backend_events_settings.h"

#include <library/cpp/unistat/unistat.h>
#include <util/string/join.h>

namespace {
    using TFieldsList = NPq2Saas::TFieldsList;
    using TProtoFieldsList = const google::protobuf::RepeatedPtrField<NPq2SaasProto::TEventLogField>;

    TFieldsList FromProtoList(const TProtoFieldsList& fields) {
        TFieldsList res;
        res.reserve(fields.size());
        for (auto&& v : fields) {
            const TVector<TString> sourceFilter(v.GetSourceFilter().begin(),
                                                v.GetSourceFilter().end());
            res.push_back({v.GetKey(), v.GetResult(), sourceFilter});
        }
        return res;
    }

    TString ToString(const TFieldsList& list) {
        TStringStream ss;
        for (size_t i = 0; i < list.size(); ++i) {
            const auto& v = list[i];
            ss << v.Key << " : " << v.Result << " : [" << JoinSeq(", ", v.SourceFilter) << "]";
            if (i < (list.size() - 1)) {
                ss << ", ";
            }
        }
        return ss.Str();
    }
}  // namespace

namespace NPq2Saas {
    TString BackendEventsSendSignalName(const TString& deliveryName) {
        return deliveryName + "-send-event";
    }

    void AddItemProperties(const TFieldsList& fields, const THashMap<TString, TString>& item,
                           NRTLine::TDocument& document, const TString& requestId) {
        for (auto&& field : fields) {
            auto it = item.find(field.Key);
            if (it != item.end()) {
                if (field.SourceFilter.empty() ||
                    Find(field.SourceFilter, item.at("source")) != field.SourceFilter.end()) {
                    document.AddProperty(field.Result, it->second);
                }
            } else {
                DEBUG_LOG << "Missing \"" << field.Key << "\" value for request "
                          << requestId << Endl;
            }
        }
    }

    void AddJsonProperties(const TFieldsList& fields, const TString& source,
                           const NJson::TJsonValue& json, NRTLine::TDocument& document,
                           const TString& requestId) {
        for (auto&& field : fields) {
            const auto value = json.GetValueByPath(field.Key);
            if (value) {
                if (field.SourceFilter.empty() ||
                    Find(field.SourceFilter, source) != field.SourceFilter.end()) {
                    document.AddProperty(field.Result, value->GetStringRobust());
                }
            } else {
                DEBUG_LOG << "Missing \"" << field.Key << "\" value for request "
                          << requestId << Endl;
            }
        }
    }

    void AddExperiments(const NJson::TJsonValue& jsonData, NRTLine::TDocument& document,
                        const TString& requestId) {
        const auto expJson = jsonData.GetValueByPath("headers.X-Yandex-ExpBoxes");
        if (!expJson) {
            return;
        }

        TVector<ui32> testIds;
        ui32 testId = 0;
        for (auto&& expIt : StringSplitter(expJson->GetString()).Split(';')) {
            const TStringBuf expStr = expIt;
            size_t pos = expStr.find(',');
            const TStringBuf testIdStr =
                    pos != TStringBuf::npos ? expStr.SubString(0, pos) : TStringBuf();
            if (TryFromString(testIdStr, testId)) {
                testIds.push_back(testId);
            }
        }

        if (testIds.empty()) {
            DEBUG_LOG << "Failed to parse experiments value for request " << requestId << Endl;
            return;
        }

        Sort(testIds);
        document.AddProperty("Experiments", JoinSeq(",", testIds));
    }

    void TDriveBackendEventsHandlerSettingsBase::PrintToStream(IOutputStream& stream) const {
        TBaseHandlerSettings::PrintToStream(stream);
        stream << "* DestinationName = " << DestinationName << Endl;
        stream << "* SourceRegExp = " << SourceRegExp.GetRegExpr() << Endl;
        stream << "* TTL = " << ToString(TTL) << Endl;
    }

    TDriveBackendEventsHandlerSettings::TDriveBackendEventsHandlerSettings(
            const NPq2SaasProto::THandlerSpecificConfig& config)
        : TDriveBackendEventsHandlerSettingsBase(config, config.GetDriveBackendEvents())
        , ExportFields(FromProtoList(config.GetDriveBackendEvents().GetExportFields()))
        , ExportDataFields(FromProtoList(config.GetDriveBackendEvents().GetExportDataFields()))
    {
    }

    void TDriveBackendEventsHandlerSettings::PrintToStream(IOutputStream& stream) const {
        TDriveBackendEventsHandlerSettingsBase::PrintToStream(stream);
        stream << "* ExportFields = " << ToString(ExportFields) << Endl;
        stream << "* ExportDataFields = " << ToString(ExportDataFields) << Endl;
    }

    void TDriveBackendEventsHandlerSettings::RegisterUnistatSignalHoles(
            const TString& deliveryName) const {
        TUnistat::Instance().DrillFloatHole(BackendEventsSendSignalName(deliveryName), "dmmm",
                                            NUnistat::TPriority(50));
    }

    TDriveBackendEventsHistoryHandlerSettings::TDriveBackendEventsHistoryHandlerSettings(
            const NPq2SaasProto::THandlerSpecificConfig& config)
        : TDriveBackendEventsHandlerSettingsBase(config, config.GetDriveBackendEventsHistory())
        , AuthExportFields(FromProtoList(config.GetDriveBackendEventsHistory().GetAuthExportFields()))
        , AuthExportDataFields(FromProtoList(config.GetDriveBackendEventsHistory().GetAuthExportDataFields()))
        , ResponseExportFields(FromProtoList(config.GetDriveBackendEventsHistory().GetResponseExportFields()))
    {
    }

    void TDriveBackendEventsHistoryHandlerSettings::PrintToStream(IOutputStream& stream) const {
        TDriveBackendEventsHandlerSettingsBase::PrintToStream(stream);
        stream << "* AuthExportFields = " << ToString(AuthExportFields) << Endl;
        stream << "* AuthExportDataFields = " << ToString(AuthExportDataFields) << Endl;
        stream << "* ResponseExportFields = " << ToString(ResponseExportFields) << Endl;
    }

    void TDriveBackendEventsHistoryHandlerSettings::RegisterUnistatSignalHoles(
            const TString& deliveryName) const {
        TUnistat::Instance().DrillFloatHole(BackendEventsSendSignalName(deliveryName), "dmmm",
                                            NUnistat::TPriority(50));
    }

    TBaseHandlerSettings::TFactory::TRegistrator<TDriveBackendEventsHandlerSettings>
            DriveBackendEventsHandlerSettings(
                    NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_BACKEND_EVENTS);

    TBaseHandlerSettings::TFactory::TRegistrator<TDriveBackendEventsHistoryHandlerSettings>
            DriveBackendEventsHistoryHandlerSettings(
                    NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_BACKEND_EVENTS_HISTORY);
}  // namespace NPq2Saas
