#include "handler.h"
#include "helpers.h"

#include "settings.h"

#include <drive/pq2saas/libhandling/actions/egts.h>
#include <drive/pq2saas/libhandling/actions/saas_indexing.h>

#include <drive/library/cpp/common/status.h>
#include <drive/library/cpp/trace/action.h>
#include <drive/library/cpp/yt/node/cast.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/unistat/unistat.h>
#include <library/cpp/yson/node/node_io.h>

#include <util/charset/utf8.h>

void NPq2Saas::TDriveTelematicsEventHandler::OnEvent(const THashMap<TString, TString>& item) try {
    const auto type = item.contains("type") ? item.at("type") : "data";
    if (type != "data" && type != "BLACKBOX_RECORDS") {
        return;
    }

    const auto imei = FromString<ui64>(item.at("imei"));
    const auto data = NYT::NodeFromJsonString(item.at("data"));

    const auto& settings = HandlerSettings.Get<TDriveTelematicsHandlerSettings>();
    if (imei % 100 > 100 * settings.GetFraction()) {
        DEBUG_LOG << "skip IMEI " << imei << Endl;
        return;
    }

    Singleton<NDrive::TTimelinesHelper>()->SetTagFilter(settings.GetBackendTagFilter());
    Singleton<NDrive::TTimelinesHelper>()->SetEndpoints(settings.GetBackendEndpoints());
    Singleton<NDrive::TTimelinesHelper>()->SetOptions(settings.GetTimelineOptions());
    Singleton<NDrive::TTimelinesHelper>()->SetQuota(settings.GetQuota());

    auto newTimelines = Singleton<NDrive::TTimelinesHelper>()->Get();
    auto newTimeline = newTimelines ? newTimelines->GetTimeline(imei) : nullptr;
    if (newTimeline) {
        bool skipAvailable = false;
        auto action = NDrive::CreateAction(imei, data, newTimeline, "new", skipAvailable);
        if (action) {
            NPq2SaasMonitoring::TManager::TDeliveryStatsPtr deliveryStats = MonManager.GetDeliveryStats(DeliveryName);
            THolder<TSaasIndexingAction> sender = MakeHolder<TSaasIndexingAction>(
                Now().Seconds(),
                action->GetDocument().GetTimestamp(),
                action->GetDocument().GetUrl(),
                std::move(*action),
                settings.GetDestinationName(),
                DependencyManager,
                deliveryStats
            );
            Queue->SafeAddAndOwn(std::move(sender));
        }
    }
} catch (const std::exception& e) {
    TString line = NDrive::PrintItem(item);
    ERROR_LOG << line << " triggered an exception: " << FormatExc(e) << Endl;
    throw;
}

void NPq2Saas::TDriveTelematicsStateHandler::OnEvent(const THashMap<TString, TString>& item) try {
    auto type = item.FindPtr("type");
    if (!type || !type->equal("BLACKBOX_RECORDS")) {
        return;
    }

    const auto& settings = HandlerSettings.Get<TDriveTelematicsStateHandlerSettings>();

    auto imei = FromString<ui64>(item.at("imei"));
    bool skipped = [&] {
        if (const auto& fraction = settings.GetFraction()) {
            if (imei % 10000 < *fraction * 10000) {
                return false;
            }
        }
        if (const auto& whitelist = settings.GetWhitelist(); !whitelist.empty()) {
            if (whitelist.contains(imei)) {
                return false;
            }
        }
        return settings.GetFraction() || !settings.GetWhitelist().empty();
    }();
    if (skipped) {
        return;
    }

    auto data = NYT::NodeFromJsonString(item.at("data"));
    auto records = NYT::FromNode<NDrive::TBlackboxRecords>(data);
    auto state = NDrive::TTelematicsStateHelper::Get(imei);
    state->Add(std::move(records));
    if (settings.ShouldIndexState()) {
        auto url = TStringBuilder() << "imei:" << imei << ":state";
        auto timestamp = state->GetTimestamp();
        NRTLine::TAction action;
        NRTLine::TDocument& document = action.AddDocument();
        document.AddProperty("data", state->Serialize<TString>());
        document.AddProperty("imei", imei);
        document.AddProperty("timestamp", timestamp.Seconds());
        document.SetUrl(url);
        document.SetTimestamp(timestamp.Seconds());
        document.SetDeadline(timestamp + TDuration::Hours(1));

        NPq2SaasMonitoring::TManager::TDeliveryStatsPtr deliveryStats = MonManager.GetDeliveryStats(DeliveryName);
        THolder<TSaasIndexingAction> sender = MakeHolder<TSaasIndexingAction>(
            Seconds(),
            timestamp.Seconds(),
            url,
            std::move(action),
            settings.GetDestinationName(),
            DependencyManager,
            deliveryStats
        );
        Queue->SafeAddAndOwn(std::move(sender));
    }

    NDrive::TFeaturesCalculationContext context;
    NDrive::TTelematicsEvents events;
    for (auto&& [calcer, options] : NDrive::GetDefaultThresholdDurationCounterFactors()) {
        auto value = NDrive::Calc(calcer, *state, options, &context);
        if (!value.Ranges.empty()) {
            auto timestamp = value.Ranges.back().first;
            NDrive::TTelematicsEvent ev;
            ev.Location = state->GetLocations().GetValueAt(timestamp);
            ev.Timestamp = timestamp;
            ev.Type = NDrive::TTelematicsEvent::EType::Aggression;
            events.push_back(std::move(ev));
        }
    }
    if (!events.empty()) {
        NDrive::TTelematicsStatePusher::Send(imei, events);
    }
} catch (const std::exception& e) {
    TString line = NDrive::PrintItem(item);
    ERROR_LOG << line << " triggered an exception: " << FormatExc(e) << Endl;
    throw;
}

namespace {

    void UpdateCarStateAsnByStatus(NPq2Saas::TEGTSAction::TState& state, NDrive::ECarStatus status) {
        state.Asn = true;
        switch (status) {
        case NDrive::ECarStatus::csFree:
        case NDrive::ECarStatus::csPost:
            state.AsnBusy = false;
            state.AsnService = false;
            break;
        case NDrive::ECarStatus::csRide:
            state.AsnBusy = true;
            state.AsnService = false;
            break;
        case NDrive::ECarStatus::csReservation:
        case NDrive::ECarStatus::csReservationPaid:
        case NDrive::ECarStatus::csAcceptance:
        case NDrive::ECarStatus::csParking:
            state.AsnBusy = false;
            state.AsnService = true;
            break;
        case NDrive::ECarStatus::csNew:
        case NDrive::ECarStatus::csService:
        case NDrive::ECarStatus::csFueling:
        case NDrive::ECarStatus::csUnknown:
            state.AsnBusy = true;
            state.AsnService = true;
            break;
        }
    }

}

void NPq2Saas::TDriveEGTSEventHandler::OnEvent(const THashMap<TString, TString>& item) try {
    const auto type = item.contains("type") ? item.at("type") : "data";
    if (type != "data" && type != "BLACKBOX_RECORDS") {
        return;
    }

    const auto imei = FromString<ui64>(item.at("imei"));
    const auto data = NYT::NodeFromJsonString(item.at("data"));

    const auto& settings = HandlerSettings.Get<TDriveEGTSHandlerSettings>();
    if (imei % 100 > 100 * settings.GetFraction()) {
        DEBUG_LOG << "skip IMEI " << imei << Endl;
        return;
    }
    const auto& modelsWhitelist = settings.GetModelsWhitelist();
    const auto& statusWhitelist = settings.GetStatusWhitelist();
    const auto& serviceTagsWhitelist = settings.GetServiceTagsWhitelist();
    const auto& tagsWhitelist = settings.GetTagsWhitelist();
    const auto& tagsBlacklist = settings.GetTagsBlacklist();

    auto timestamp = NDrive::GetTimestamp(data);
    if (settings.GetMaxAge()) {
        auto age = Now() - timestamp;
        if (age.Seconds() > settings.GetMaxAge()) {
            INFO_LOG << "Too old data for " << imei << " " << timestamp.Seconds() << Endl;
            return;
        }
    }

    if (!Singleton<NDrive::TTimelinesHelper>()->Get()) {
        TSet<TString> trackedTags;
        trackedTags.insert(tagsWhitelist.begin(), tagsWhitelist.end());
        trackedTags.insert(tagsBlacklist.begin(), tagsBlacklist.end());
        trackedTags.insert(serviceTagsWhitelist.begin(), serviceTagsWhitelist.end());
        Singleton<NDrive::TTimelinesHelper>()->SetTrackedTags(trackedTags);
    }

    Singleton<NDrive::TTimelinesHelper>()->SetHostPort(settings.GetFrontendApiHostPort(), settings.GetFrontendApiExtraCgi());
    Singleton<NDrive::TTimelinesHelper>()->SetOptions(settings.GetTimelineOptions());
    Singleton<NDrive::TTimelinesHelper>()->SetQuota(settings.GetQuota());

    NDrive::ECarStatus newCarStatus = NDrive::ECarStatus::csFree;
    auto newTimelines = Singleton<NDrive::TTimelinesHelper>()->Get();
    if (newTimelines) {
        auto newTimeline = newTimelines->GetTimeline(imei);
        const auto& car = newTimeline->GetCar();
        const auto status = newTimeline->GetNativeStatus();
        if (!modelsWhitelist.empty() && !modelsWhitelist.contains(car.Model)) {
            INFO_LOG << "skipping " << imei << " by models whitelist: " << car.Model << Endl;
            return;
        }
        if (!statusWhitelist.empty() && status && !statusWhitelist.contains(status)) {
            INFO_LOG << "skipping " << imei << " by status whitelist: " << status << Endl;
            return;
        }
        if (!tagsWhitelist.empty()) {
            bool found = false;
            for (auto&& tag : tagsWhitelist) {
                if (car.GetTag(tag)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                INFO_LOG << "skipping " << imei << " by tags whitelist: " << JoinStrings(tagsWhitelist.begin(), tagsWhitelist.end(), ",") << Endl;
                return;
            }
        }
        if (!tagsBlacklist.empty()) {
            for (auto&& tag : tagsBlacklist) {
                if (car.GetTag(tag)) {
                    INFO_LOG << "skipping " << imei << " by tags blacklist: " << tag << Endl;
                    return;
                }
            }
        }

        auto newStatus = newTimeline->GetStatus(timestamp);
        auto servicePerformers = car.GetServicePerformers(&serviceTagsWhitelist);
        if (!servicePerformers.empty()) {
            if (servicePerformers.contains(newStatus.GetUserId())) {
                INFO_LOG << "skipping " << newStatus << " by serviceman" << Endl;
                return;
            }
        }
        newCarStatus = NDrive::GetStatus(newStatus.GetCarStatus());
    }

    NPq2SaasMonitoring::TManager::TDeliveryStatsPtr deliveryStats = MonManager.GetDeliveryStats(DeliveryName);
    NPq2Saas::TEGTSAction action(ToString(imei), settings.GetDestinationName(), DependencyManager, deliveryStats);
    NPq2Saas::TEGTSAction::TState state;
    state.Timestamp = NDrive::GetTimestamp(data);
    state.Latitude = NDrive::GetLatitude(data);
    state.Longitude = NDrive::GetLongitude(data);
    state.Speed = NDrive::GetSpeed(data);
    state.Direction = NDrive::GetCourse(data);
    state.Free = NDrive::IsFree(newCarStatus);
    if (settings.GetSendExtendedStatus()) {
        UpdateCarStateAsnByStatus(state, newCarStatus);
    }
    state.Moving = newCarStatus == NDrive::ECarStatus::csRide;
    action.Send(state);
    TUnistat::Instance().PushSignalUnsafe(DeliveryName + "-sent", 1);
} catch (const std::exception& e) {
    TString line = NDrive::PrintItem(item);
    ERROR_LOG << line << " triggered an exception: " << FormatExc(e) << Endl;
    throw;
}

void NPq2Saas::TDriveMobileMetrikaHandler::OnEvent(const THashMap<TString, TString>& item) try {
    const auto& settings = HandlerSettings.Get<TDriveMobileMetrikaHandlerSettings>();
    auto& helper = NDrive::TMobileMetrikaHelper::Instance();
    helper.SetHostPort(settings.GetFrontendApiHostPort(), {});
    helper.Update();

    const TString& deviceId = item.at("DeviceID");
    const TString& networkInterfacesString = item.contains("NetworksInterfaces_Macs") ? item.at("NetworksInterfaces_Macs") : Default<TString>();

    TVector<TString> macs;
    if (networkInterfacesString) {
        NJson::TJsonValue j = NJson::ReadJsonFastTree(networkInterfacesString);
        Y_ENSURE(j.IsArray(), "NetworksInterfaces_Macs should be a Json array");
        for (auto&& i : j.GetArray()) {
            macs.push_back(i.GetStringRobust());
        }
    }
    TString headId = !macs.empty() ? macs.front() : Default<TString>();
    headId = ToLowerUTF8(headId);
    if (deviceId && headId) {
        helper.RegisterDeviceId(deviceId, headId);
    }

    if (!item.contains("Latitude") || !item.contains("Longitude")) {
        DEBUG_LOG << "skip DeviceId " << deviceId << Endl;
        return;
    }

    const auto latitude = FromString<double>(item.at("Latitude"));
    const auto longitude = FromString<double>(item.at("Longitude"));

    TInstant timestamp;
    auto locationTimestamp = item.find("LocationTimestamp");
    auto eventTiemstamp = item.find("EventTimestamp");
    if (locationTimestamp != item.end()) {
        timestamp = TInstant::Seconds(FromString<ui64>(locationTimestamp->second));
    } else if (eventTiemstamp != item.end()) {
        timestamp = TInstant::Seconds(FromString<ui64>(eventTiemstamp->second));
    }

    auto car = helper.GetCar(deviceId);
    if (!car) {
        DEBUG_LOG << "skip DeviceId " << deviceId << Endl;
        return;
    }

    NDrive::TLocation location;
    location.Latitude = latitude;
    location.Longitude = longitude;
    location.Since = timestamp;
    location.Timestamp = timestamp;
    location.Type = NDrive::TLocation::External;
    location.Name = "head";

    NPq2SaasMonitoring::TManager::TDeliveryStatsPtr deliveryStats = MonManager.GetDeliveryStats(DeliveryName);
    for (auto&& destination : settings.GetDestinationNames()) {
        NRTLine::TAction action = NDrive::TPusher::CreateAction(ToString(car->IMEI), location);
        auto sender = MakeHolder<TSaasIndexingAction>(
            Now().Seconds(),
            action.GetDocument().GetTimestamp(),
            action.GetDocument().GetUrl(),
            std::move(action),
            destination,
            DependencyManager,
            deliveryStats
        );
        Queue->SafeAddAndOwn(THolder(sender.Release()));
    }
} catch (const std::exception& e) {
    TString line = NDrive::PrintItem(item);
    ERROR_LOG << line << " triggered an exception: " << FormatExc(e) << Endl;
    throw;
}

void NPq2Saas::PrefetchTimelines(const TDriveTelematicsHandlerBaseSettings& settings) {
    TSet<TString> trackedTags;
    trackedTags.insert(settings.GetTagsWhitelist().begin(), settings.GetTagsWhitelist().end());
    trackedTags.insert(settings.GetTagsBlacklist().begin(), settings.GetTagsBlacklist().end());
    if (!trackedTags.empty()) {
        Singleton<NDrive::TTimelinesHelper>()->SetTrackedTags(trackedTags);
    }
    if (settings.GetFrontendApiHostPort()) {
        Singleton<NDrive::TTimelinesHelper>()->SetHostPort(settings.GetFrontendApiHostPort(), settings.GetFrontendApiExtraCgi());
    } else {
        Singleton<NDrive::TTimelinesHelper>()->SetEndpoints(settings.GetBackendEndpoints());
    }
    Singleton<NDrive::TTimelinesHelper>()->SetOptions(settings.GetTimelineOptions());
    Singleton<NDrive::TTimelinesHelper>()->SetQuota(settings.GetQuota());
    Singleton<NDrive::TTimelinesHelper>()->Get();
}

NPq2Saas::IEventHandler::TFactory::TRegistrator<NPq2Saas::TDriveMobileMetrikaHandler> DriveMobileMetrika(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_MOBILE_METRIKA);
NPq2Saas::IEventHandler::TFactory::TRegistrator<NPq2Saas::TDriveTelematicsEventHandler> DriveTelematics(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_TELEMATICS);
NPq2Saas::IEventHandler::TFactory::TRegistrator<NPq2Saas::TDriveTelematicsStateHandler> DriveTelematicsState(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_TELEMATICS_STATE);
NPq2Saas::IEventHandler::TFactory::TRegistrator<NPq2Saas::TDriveEGTSEventHandler> DriveEGTS(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_EGTS);
