#pragma once

#include <drive/library/cpp/auth/tvm.h>
#include <drive/library/cpp/threading/concurrent_hash.h>
#include <drive/library/cpp/timeline/meta.h>
#include <drive/library/cpp/timeline/timeline.h>
#include <drive/library/cpp/tvm/logger.h>

#include <drive/telematics/server/data/blackbox.h>
#include <drive/telematics/server/data/events.h>
#include <drive/telematics/server/data/factors.h>
#include <drive/telematics/server/location/location.h>
#include <drive/telematics/server/pusher/pusher.h>

#include <rtline/library/json/cast.h>
#include <rtline/util/network/neh.h>

#include <util/system/env.h>

namespace NClickHouse {
    struct TClientOptions;
}

namespace NDrive {
    class TClickHousePusher;

    class TMobileMetrikaHelper {
    public:
        static TMobileMetrikaHelper& Instance() {
            return *Singleton<TMobileMetrikaHelper>();
        }

    public:
        TMaybe<NDrive::TSessionRequester::TCar> GetCar(const TString& deviceId) const {
            if (!deviceId) {
                return {};
            }

            TString headId;
            {
                TReadGuard guard(DeviceId2HeadIdLock);
                auto p = DeviceId2HeadId.find(deviceId);
                if (p != DeviceId2HeadId.end()) {
                    headId = p->second;
                }
            }
            if (!headId) {
                return {};
            }

            TString carId;
            {
                TReadGuard guard(HeadId2CarIdLock);
                auto p = HeadId2CarId.find(headId);
                if (p != HeadId2CarId.end()) {
                    carId = p->second;
                }
            }
            if (!carId) {
                return {};
            }

            {
                TReadGuard guard(CarId2CarLock);
                auto p = CarId2Car.find(carId);
                if (p != CarId2Car.end()) {
                    return p->second;
                } else {
                    return {};
                }
            }
        }

        void SetHostPort(const TString& value, const TString& extraCgi) {
            if (!value) {
                return;
            }
            if (Requester) {
                return;
            }

            auto guard = Guard(Lock);
            if (!Requester) {
                auto token = GetEnv("DRIVE_TOKEN");
                auto requester = MakeHolder<NDrive::TSessionRequester>(value, extraCgi, token);
                Update(*requester);
                Requester = std::move(requester);
            }
        }

        void RegisterDeviceId(const TString& deviceId, const TString& headId) {
            Y_ENSURE(deviceId);
            Y_ENSURE(headId);
            {
                TReadGuard guard(DeviceId2HeadIdLock);
                auto p = DeviceId2HeadId.find(deviceId);
                if (p != DeviceId2HeadId.end() && p->second == headId) {
                    return;
                }
            }
            {
                TWriteGuard guard(DeviceId2HeadIdLock);
                DeviceId2HeadId[deviceId] = headId;
            }
        }

        void Update() {
            TTryGuard tg(Lock);
            if (!tg) {
                return;
            }
            if (!Requester) {
                return;
            }
            Update(*Requester);
        }

    private:
        void Update(const NDrive::TSessionRequester& requester) {
            auto now = Now();
            if (now < CarsUpdate + CarsUpdateInterval) {
                return;
            }

            auto cars = requester.GetCars();
            auto heads = requester.GetHeadId({});
            for (auto&&[id, car] : cars) {
                {
                    TReadGuard guard(CarId2CarLock);
                    auto p = CarId2Car.find(id);
                    if (p != CarId2Car.end() && p->second.IMEI == car.IMEI) {
                        continue;
                    }
                }
                {
                    TWriteGuard guard(CarId2CarLock);
                    CarId2Car[id] = car;
                    INFO_LOG << "setting car_id: " << id << " imei: " << car.IMEI << Endl;
                }
            }
            for (auto&&[carId, headId] : heads) {
                {
                    TReadGuard guard(HeadId2CarIdLock);
                    auto p = HeadId2CarId.find(headId);
                    if (p != HeadId2CarId.end() && p->second == carId) {
                        continue;
                    }
                }
                {
                    TWriteGuard guard(HeadId2CarIdLock);
                    HeadId2CarId[headId] = carId;
                    INFO_LOG << "setting head_id: " << headId << " car_id: " << carId << Endl;
                }
            }
            CarsUpdate = now;
        }

    private:
        THolder<NDrive::TSessionRequester> Requester;
        TInstant CarsUpdate = TInstant::Zero();
        TDuration CarsUpdateInterval = TDuration::Minutes(10);
        TMutex Lock;

        TMap<TString, TString> DeviceId2HeadId;
        TRWMutex DeviceId2HeadIdLock;

        TMap<TString, TString> HeadId2CarId;
        TRWMutex HeadId2CarIdLock;

        TMap<TString, NDrive::TSessionRequester::TCar> CarId2Car;
        TRWMutex CarId2CarLock;
    };

    class TTimelinesHelper {
    public:
        struct TEndpoint {
            TString HostPort;
            TString ExtraCgi;

            inline TEndpoint(const TString& hostPort, const TString& extraCgi)
                : HostPort(hostPort)
                , ExtraCgi(extraCgi)
            {
            }
            TEndpoint(TStringBuf endpoint);
        };
        using TEndpoints = TVector<TEndpoint>;

    public:
        static TTimelinesHelper& Instance();

    public:
        void SetHostPort(const TString& value, const TString& extraCgi) {
            if (!value) {
                return;
            }
            TEndpoints endpoints;
            endpoints.emplace_back(value, extraCgi);
            SetEndpoints(endpoints);
        }

        template <class T>
        void SetEndpoints(const T& endpoints) {
            auto guard = Guard(Lock);
            if (!Timelines) {
                Endpoints.clear();
                for (auto&& endpoint : endpoints) {
                    Endpoints.emplace_back(endpoint);
                }
                Token = GetEnv("DRIVE_TOKEN");
                VERIFY_WITH_LOG(Token, "DRIVE_TOKEN is missing");

                TVector<NDrive::TRequesterPtr> requesters;
                for (auto&& endpoint : Endpoints) {
                    auto requester = MakeAtomicShared<NDrive::TSessionRequester>(endpoint.HostPort, endpoint.ExtraCgi, Token, Quota);
                    if (TrackedTags) {
                        requester->SetTagFilter(TagFilter);
                        requester->SetTrackedTags(*TrackedTags);
                        requester->SetTrackSpecialTags(true);
                    }
                    requesters.push_back(std::move(requester));
                }

                NDrive::TRequesterPtr requester;
                if (requesters.size() == 1) {
                    requester = requesters.front();
                } else {
                    requester = MakeAtomicShared<NDrive::TMetaRequester>(std::move(requesters));
                }

                Timelines = MakeHolder<NDrive::TNewTimelines>(
                    requester, Options
                );
            }
        }

        void SetOptions(const NDrive::TNewTimelines::TOptions& value) {
            Options = value;
        }

        void SetTagFilter(const TString& value) {
            TagFilter = value;
        }

        template <class T>
        void SetTrackedTags(T&& tags) {
            TrackedTags = std::forward<T>(tags);
        }

        void SetQuota(ui64 value) {
            Quota = value;
        }

        NDrive::TNewTimelines* Get() {
            auto guard = Guard(Lock);
            return Timelines.Get();
        }

    private:
        TEndpoints Endpoints;
        TMutex Lock;
        TString Token;
        NDrive::TNewTimelines::TOptions Options;
        TString TagFilter;
        TMaybe<TSet<TString>> TrackedTags;
        ui64 Quota = 0;
        THolder<NDrive::TNewTimelines> Timelines;
    };

    class TTelematicsHistoryHelper {
    public:
        static const TClickHousePusher& GetClient() {
            return Instance().GetClientImpl();
        }
        static void SetClientOptions(const TVector<NClickHouse::TClientOptions>& options) {
            Instance().SetClientOptionsImpl(options);
        }
        static void SetSuccessBackoff(TDuration value) {
            Instance().SuccessBackoff = value;
        }
        static void SetFailureBackoff(TDuration value) {
            Instance().FailureBackoff = value;
        }

    public:
        ~TTelematicsHistoryHelper();

    private:
        static TTelematicsHistoryHelper& Instance();

    private:
        const TClickHousePusher& GetClientImpl() const;
        void SetClientOptionsImpl(const TVector<NClickHouse::TClientOptions>& options);

    private:
        THolder<TClickHousePusher> Client;

        TDuration SuccessBackoff;
        TDuration FailureBackoff;
    };

    class TTelematicsStateHelper {
    public:
        static NDrive::TTelematicsHistoryPtr Get(ui64 imei) {
            return Singleton<TTelematicsStateHelper>()->GetImpl(imei);
        }
        static void SetLocationDepth(TDuration value) {
            Singleton<TTelematicsStateHelper>()->LocationDepth = value;
        }
        static void SetSensorDepth(ui32 value) {
            Singleton<TTelematicsStateHelper>()->SensorDepth = value;
        }

    private:
        NDrive::TTelematicsHistoryPtr GetImpl(ui64 imei) {
            NDrive::TTelematicsHistoryPtr result;
            if (!result) {
                auto p = HistoryMap.find(imei);
                if (p) {
                    result = p->second;
                }
            }
            {
                auto history = MakeAtomicShared<NDrive::TTelematicsHistory>(LocationDepth, SensorDepth);
                auto&& [pp, inserted] = HistoryMap.emplace(imei, std::move(history));
                result = pp->second;
            }
            CHECK_WITH_LOG(result) << imei;
            return result;
        }

    private:
        NUtil::TConcurrentHashMap<ui64, NDrive::TTelematicsHistoryPtr> HistoryMap;
        TDuration LocationDepth = TDuration::Minutes(10);
        ui32 SensorDepth = 1 << 10;
    };

    class TTelematicsStatePusher {
    public:
        static void Send(ui64 imei, const NDrive::TTelematicsEvents& events) {
            return Singleton<TTelematicsStatePusher>()->SendImpl(imei, events);
        }

        static constexpr ui32 SelfClientId = 2015669;
        static constexpr ui32 DestinationId = 2000615;

    public:
        TTelematicsStatePusher()
            : Client("https://leasing-cabinet.carsharing.yandex.net")
            , Tvm([] {
                NTvmAuth::NTvmApi::TClientSettings settings;
                settings.SetSelfTvmId(SelfClientId);
                settings.EnableServiceTicketsFetchOptions(GetEnv("TVM_SECRET"), { DestinationId });
                return MakeAtomicShared<NTvmAuth::TTvmClient>(settings, CreateGlobalTvmLogger());
            }(), DestinationId)
        {
        }

    private:
        void SendImpl(ui64 imei, const NDrive::TTelematicsEvents& events) {
            NNeh::THttpRequest request;
            request.SetUri("/api/telematics/event/push");
            request.SetCgiData("&backend_cluster=prestable");
            NJson::TJsonValue post;
            post["imei"] = ToString(imei);
            post["events"] = NJson::ToJson(events);
            request.SetPostData(post.GetStringRobust());
            Tvm.UpdateRequest(request);
            Client.SendAsync(request);
        }

    private:
        NNeh::THttpClient Client;
        NDrive::TTvmAuth Tvm;
    };

    inline TString PrintItem(const THashMap<TString, TString>& item) {
        TString line;
        for (auto&& i : item) {
            line.append(i.first);
            line.append('=');
            line.append(i.second);
            line.append('\t');
        }
        return line;
    }
}
