#include "sensors_history.h"
#include "helpers.h"

#include <drive/library/cpp/yt/node/cast.h>

#include <drive/telematics/server/data/clickhouse/client.h>
#include <drive/telematics/server/sensors/calculator.h>
#include <drive/telematics/server/sensors/validation.h>

#include <library/cpp/clickhouse/client/client.h>
#include <library/cpp/yson/node/node_io.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/url/url.h>

#include <util/charset/utf8.h>
#include <util/system/env.h>

namespace NPq2Saas {
    void TDriveTelematicsHistorySettings::Verify(TDependencyManagerPtr) const {
        {
            NDrive::TNewTimelines::TOptions options;
            options.UpdateLocalDepth = TDuration::Zero();
            NDrive::TTimelinesHelper::Instance().SetEndpoints(BackendEndpoints);
            NDrive::TTimelinesHelper::Instance().SetOptions(options);
            NDrive::TTimelinesHelper::Instance().Get();
        }
        {
            NClickHouse::TClientOptions options;
            options.SetHost(ClickHouseHost);
            options.SetPort(ClickHousePort);
            options.SetDefaultDatabase(ClickHouseDatabase);
            options.SetUser(ClickHouseUser);
            options.SetPassword(GetEnv("CLICKHOUSE_PASSWORD"));
            options.SetUseSsl(true);
            TVector<NClickHouse::TClientOptions> clientOptions;
            for (auto&& endpoint : ClickHouseEndpoints) {
                TStringBuf scheme;
                TStringBuf host;
                ui16 port = 0;
                Y_ENSURE(TryGetSchemeHostAndPort(endpoint, scheme, host, port), "cannot TryGetSchemeHostAndPort from " << endpoint);
                options.SetHost(ToString(host));
                options.SetPort(port);
                clientOptions.push_back(options);
            }
            if (ClickHouseHost) {
                options.SetHost(ClickHouseHost);
                options.SetPort(ClickHousePort);
                clientOptions.push_back(options);
            }
            NDrive::TTelematicsHistoryHelper::SetSuccessBackoff(SuccessBackoff);
            NDrive::TTelematicsHistoryHelper::SetFailureBackoff(FailureBackoff);
            NDrive::TTelematicsHistoryHelper::SetClientOptions(clientOptions);
        }
        {
            NDrive::TTelematicsStateHelper::SetLocationDepth(LocationDepth);
            NDrive::TTelematicsStateHelper::SetSensorDepth(SensorDepth);
        }
    }

    void TDriveTelematicsHistoryEventHandler::OnEvent(const THashMap<TString, TString>& item) try {
        const auto type = item.contains("type") ? item.at("type") : "data";
        if (type != "data" && type != "BLACKBOX_RECORDS") {
            return;
        }

        NDrive::TSensorCalculator calculator;

        const auto imei = item.at("imei");
        const auto integerImei = FromStringWithDefault<ui64>(imei, 0);
        const auto data = NYT::NodeFromJsonString(item.at("data"));
        auto records = NYT::FromNode<NDrive::TBlackboxRecords>(data);
        auto received = TInstant::Seconds(FromString<ui64>(item.at("unixtime")));

        const auto& settings = HandlerSettings.Get<TDriveTelematicsHistorySettings>();
        const auto& client = NDrive::TTelematicsHistoryHelper::GetClient();
        const auto timelines = NDrive::TTimelinesHelper::Instance().Get();
        const auto timeline = Yensured(timelines)->GetTimeline(integerImei);

        const auto& objectId = timeline ? timeline->GetCar().Id : Default<TString>();
        const auto state = NDrive::TTelematicsStateHelper::Get(integerImei);
        for (auto&& record : records.Values) {
            auto pushLocation = [&](NDrive::TGpsLocation&& location) {
                {
                    auto full = NDrive::TLocation(location);
                    auto [l, status] = NDrive::Validate(std::move(full), state->GetSensors());
                    if (status != NDrive::EDataValidationStatus::Ok && status != NDrive::EDataValidationStatus::ZeroValue) {
                        DEBUG_LOG << imei << ": validation error for " << l.ToJson().GetStringRobust() << ": " << status << Endl;
                        return;
                    }
                }
                bool pushed = false;
                if (!pushed) {
                    if (location.Speed) {
                        pushed = true;
                    }
                }
                if (!pushed) {
                    auto lastTimestamp = state->GetLocations().GetTimestamp();
                    if (lastTimestamp > location.Timestamp.Get()) {
                        pushed = true;
                    }
                }
                if (!pushed) {
                    auto previous = state->GetLocations().GetValueAt(location.Timestamp, /*skipZeros=*/false);
                    if (!previous || std::abs(previous->Latitude - location.Latitude) > 0.001 || std::abs(previous->Longitude - location.Longitude) > 0.001) {
                        pushed = true;
                    }
                }
                if (!pushed) {
                    auto lastPush = state->GetPushed(location).GetOrElse({});
                    if (lastPush + settings.IdleTime < location.Timestamp.Get()) {
                        pushed = true;
                    }
                }
                if (pushed) {
                    client.Push(objectId, imei, location, received);
                    state->OnPush(location);
                }
                state->Add(std::move(location));
            };
            if (auto location = record.Location; location && settings.EnableLocations) {
                pushLocation(std::move(*location));
            }

            auto push = [&](NDrive::TSensor&& sensor) {
                if (settings.EnableCurrentSensors) {
                    NDrive::TSensorClickHouseRecord record(sensor, imei, objectId);
                    client.Push(std::move(record));
                }
                if (!settings.SensorWhitelist.contains(sensor)) {
                    return;
                }
                {
                    auto [s, status] = NDrive::Validate(std::move(sensor), state->GetSensors());
                    if (status != NDrive::EDataValidationStatus::Ok) {
                        DEBUG_LOG << imei << ": validation error for " << s.ToJson().GetStringRobust() << ": " << status << Endl;
                        return;
                    }
                    sensor = std::move(s);
                }
                bool pushed = false;
                if (!pushed) {
                    auto lastTimestamp = state->GetSensors().GetTimestamp(sensor);
                    if (lastTimestamp > sensor.Timestamp.Get()) {
                        pushed = true;
                    }
                }
                if (!pushed) {
                    auto previous = state->GetSensors().Get(sensor.Id, sensor.SubId, sensor.Timestamp);
                    if (!previous || previous->Value != sensor.Value) {
                        pushed = true;
                    }
                }
                if (!pushed) {
                    auto lastPush = state->GetPushed(sensor).GetOrElse({});
                    if (lastPush + settings.IdleTime < sensor.Timestamp.Get()) {
                        pushed = true;
                    }
                }
                if (pushed) {
                    auto groupstamp = TInstant::Hours(sensor.Timestamp.Get().Hours());
                    client.Push(objectId, imei, sensor, groupstamp, received);
                    state->OnPush(sensor);
                }
                state->Add(std::move(sensor));
            };
            if (!settings.EnableSensors) {
                continue;
            }
            for (auto&& sensor : record.Sensors) {
                auto derived = calculator.Derive(sensor, Yensured(state)->GetSensors());
                for (auto&& derivedSensor : derived) {
                    push(std::move(derivedSensor));
                }
                push(std::move(sensor));
            }
        }
    } catch (const std::exception& e) {
        TString line = NDrive::PrintItem(item);
        ERROR_LOG << line << " triggered an exception: " << FormatExc(e) << Endl;
        throw;
    }

    TBaseHandlerSettings::TFactory::TRegistrator<TDriveTelematicsHistorySettings> TDriveTelematicsHistorySettings::Registrator(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_TELEMATICS_HISTORY);
    IEventHandler::TFactory::TRegistrator<TDriveTelematicsHistoryEventHandler> TDriveTelematicsHistoryEventHandler::Registrator(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_TELEMATICS_HISTORY);
}
