#pragma once

#include <drive/pq2saas/libhandling/interface/interface.h>
#include <drive/pq2saas/libhandling/interface/settings.h>

#include <drive/telematics/protocol/vega.h>

namespace NPq2Saas {
    class TDriveTelematicsHistorySettings : public TBaseHandlerSettings {
    public:
        TDriveTelematicsHistorySettings(const NPq2SaasProto::THandlerSpecificConfig& config)
            : TBaseHandlerSettings(config)
        {
            const auto& dth = config.GetDriveTelematicsHistory();
            for (auto&& endpoint : dth.GetBackendEndpoint()) {
                BackendEndpoints.push_back(endpoint);
            }
            for (auto&& endpoint : dth.GetClickHouseEndpoint()) {
                ClickHouseEndpoints.push_back(endpoint);
            }
            if (dth.HasClickHouseDatabase()) {
                ClickHouseDatabase = dth.GetClickHouseDatabase();
            }
            if (dth.HasClickHouseHost()) {
                ClickHouseHost = dth.GetClickHouseHost();
            }
            if (dth.HasClickHouseUser()) {
                ClickHouseUser = dth.GetClickHouseUser();
            }
            if (dth.HasClickHousePort()) {
                ClickHousePort = dth.GetClickHousePort();
            }
            if (dth.HasEnableCurrentSensors()) {
                EnableCurrentSensors = dth.GetEnableCurrentSensors();
            }
            if (dth.HasEnableLocations()) {
                EnableLocations = dth.GetEnableLocations();
            }
            if (dth.HasEnableSensors()) {
                EnableSensors = dth.GetEnableSensors();
            }
            if (!(dth.GetSensorWhitelist()).empty()) {
                SensorWhitelist.clear();
                for (auto&& sensorId : dth.GetSensorWhitelist()) {
                    SensorWhitelist.insert(sensorId);
                }
            }
            if (dth.HasSuccessBackoff()) {
                SuccessBackoff = TDuration::Seconds(dth.GetSuccessBackoff());
            }
            if (dth.HasFailureBackoff()) {
                FailureBackoff = TDuration::Seconds(dth.GetFailureBackoff());
            }
        }

        TVector<THandlerDependency> GetDependencies() const override {
            TVector<THandlerDependency> result;
            return result;
        }

        void PrintToStream(IOutputStream& stream) const override {
            stream << "BackendEndpoints : " << JoinRange(",", BackendEndpoints.begin(), BackendEndpoints.end()) << Endl;
            stream << "ClickHouseEndpoints : " << JoinRange(",", ClickHouseEndpoints.begin(), ClickHouseEndpoints.end()) << Endl;
            stream << "ClickHouseDatabase : " << ClickHouseDatabase << Endl;
            stream << "ClickHouseHost : " << ClickHouseHost << Endl;
            stream << "ClickHousePort : " << ClickHousePort << Endl;
            stream << "ClickHouseUser : " << ClickHouseUser << Endl;
            stream << "EnableCurrentSensors: " << EnableCurrentSensors << Endl;
            stream << "EnableLocations : " << EnableLocations << Endl;
            stream << "EnableSensors : " << EnableSensors << Endl;
        }

        void Verify(TDependencyManagerPtr) const override;

    public:
        TVector<TString> BackendEndpoints;
        TVector<TString> ClickHouseEndpoints;
        TString ClickHouseDatabase = "production";
        TString ClickHouseHost;
        TString ClickHouseUser;
        ui16 ClickHousePort = 9440;
        bool EnableCurrentSensors = false;
        bool EnableLocations = false;
        bool EnableSensors = true;
        TDuration IdleTime = TDuration::Seconds(300);
        TDuration LocationDepth = TDuration::Seconds(10);
        TDuration SuccessBackoff = TDuration::Seconds(10);
        TDuration FailureBackoff = TDuration::Seconds(20);
        ui32 SensorDepth = 8;

        TSet<NDrive::TSensorId> SensorWhitelist = {
            VEGA_NRF_VISIBLE_MARKS_BF,
            CAN_ENGINE_IS_ON,
            CAN_ENGINE_TEMP,
            CAN_FUEL_LEVEL_L,
            CAN_FUEL_LEVEL_P,
            CAN_ODOMETER_KM,
            CAN_DRIVER_DOOR,
            CAN_PASS_DOOR,
            CAN_L_REAR_DOOR,
            CAN_R_REAR_DOOR,
            CAN_HOOD,
            CAN_TRUNK,
            VEGA_GPS_MOVE_SENSOR,
            VEGA_POWER_VOLTAGE,
            VEGA_ACC_VOLTAGE,
            VEGA_ECM_DTC_LIST_SENSOR,
            { CAN_CHECK_OIL, CAN_ENGINE_IS_ON },
            { CAN_CHECK_COOLANT, CAN_ENGINE_IS_ON },
            { CAN_AIRBAG, CAN_ENGINE_IS_ON },
            { CAN_CHECK_ENGINE, CAN_ENGINE_IS_ON },
            { CAN_FAULT_LIGHTING, CAN_ENGINE_IS_ON },
            { CAN_INFLATION_PRESSURE, CAN_ENGINE_IS_ON },
            { CAN_ESP, CAN_ENGINE_IS_ON },
            { CAN_WARNING, CAN_ENGINE_IS_ON },
            { VEGA_CAN_WASHER_LIQUID, CAN_ENGINE_IS_ON },
            { CAN_CHECK_BRAKE_PADS, CAN_ENGINE_IS_ON },
            { VEGA_GSENSOR_AXIS_X, CAN_ENGINE_IS_ON },
            { VEGA_GSENSOR_AXIS_Y, CAN_ENGINE_IS_ON },
            { VEGA_GSENSOR_AXIS_Z, CAN_ENGINE_IS_ON },
            { VEGA_SETTING_SERVER_PIN, 0 },
            { VEGA_SETTING_SERVER_PIN, 1 },
            { VEGA_SETTING_SERVER_PIN, 2 },
            { VEGA_SETTING_SERVER_PIN, 3 },
            VEGA_SETTING_PIN_CODE,
            CAN_CUSTOM_FUEL_VOLUME,
            VEGA_VIN_SENSOR
        };

    private:
        static TFactory::TRegistrator<TDriveTelematicsHistorySettings> Registrator;
    };

    class TDriveTelematicsHistoryEventHandler: public IEventHandler {
    public:
        using IEventHandler::IEventHandler;

        void OnEvent(const THashMap<TString, TString>& item) override;

    private:
        static TFactory::TRegistrator<TDriveTelematicsHistoryEventHandler> Registrator;
    };
}
