#include "settings.h"
#include "handler.h"

#include <library/cpp/unistat/unistat.h>

#include <util/string/join.h>

NPq2Saas::TDriveMobileMetrikaHandlerSettings::TDriveMobileMetrikaHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
    , DestinationNames(config.GetDriveMobileMetrika().GetDestinationName().begin(), config.GetDriveMobileMetrika().GetDestinationName().end())
    , FrontendApiHostPort(config.GetDriveMobileMetrika().GetFrontendApi())
{
}

void NPq2Saas::TDriveMobileMetrikaHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* DestinationName = " << JoinStrings(DestinationNames.begin(), DestinationNames.end(), ",") << Endl;
    stream << "* FrontendApiHostPort = " << FrontendApiHostPort << Endl;
}

NPq2Saas::TDriveTelematicsHandlerBaseSettings::TDriveTelematicsHandlerBaseSettings(const NPq2SaasProto::THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
    , BackendTagFilter(config.GetDriveTelematics().GetBackendTagFilter())
    , FrontendApiHostPort(config.GetDriveTelematics().GetFrontendApi())
    , FrontendApiExtraCgi(config.GetDriveTelematics().GetFrontendApiExtraCgi())
    , DestinationName(config.GetDriveTelematics().GetDestinationName())
    , Fraction(config.GetDriveTelematics().GetFraction())
    , MaxAge(config.GetDriveTelematics().GetMaxAge())
    , Quota(config.GetDriveTelematics().GetQuota())
{
    if (FrontendApiHostPort) {
        BackendEndpoints.push_back(FrontendApiHostPort);
    }
    for (auto&& endpoint : config.GetDriveTelematics().GetBackendEndpoint()) {
        BackendEndpoints.push_back(endpoint);
    }
    for (auto&& model : config.GetDriveTelematics().GetModelsWhitelist()) {
        ModelsWhitelist.insert(model);
    }
    for (auto&& model : config.GetDriveTelematics().GetTagsWhitelist()) {
        TagsWhitelist.insert(model);
    }
    for (auto&& model : config.GetDriveTelematics().GetTagsBlacklist()) {
        TagsBlacklist.insert(model);
    }
    for (auto&& status : config.GetDriveTelematics().GetStatusWhitelist()) {
        StatusWhitelist.insert(status);
    }
    for (auto&& tag : config.GetDriveTelematics().GetServiceTagsWhitelist()) {
        ServiceTagsWhitelist.insert(tag);
    }
    const auto& proto = config.GetDriveTelematics();
    if (proto.GetUpdateInteval() > 0) {
        TimelineOptions.UpdateInterval = TDuration::Seconds(proto.GetUpdateInteval());
    }
    if (proto.GetUpdateLocalInterval() > 0) {
        TimelineOptions.UpdateLocalInterval = TDuration::Seconds(proto.GetUpdateLocalInterval());
    }
    if (proto.GetUpdateLocalDepth() > 0) {
        TimelineOptions.UpdateLocalDepth = TDuration::Seconds(proto.GetUpdateLocalDepth());
    }
    if (proto.GetUpdateGlobalDepth() > 0) {
        TimelineOptions.UpdateGlobalDepth = TDuration::Seconds(proto.GetUpdateGlobalDepth());
    }
    if (proto.GetUpdateOverlap() > 0) {
        TimelineOptions.UpdateOverlap = TDuration::Seconds(proto.GetUpdateOverlap());
    }
    if (proto.GetUpdatePauseLag() > 0) {
        TimelineOptions.UpdatePauseLag = TDuration::Seconds(proto.GetUpdatePauseLag());
    }
    if (proto.GetUpdateCarsInterval() > 0) {
        TimelineOptions.UpdateCarsInterval = TDuration::Seconds(proto.GetUpdateCarsInterval());
    }
    if (proto.GetUpdateRealtimePause() > 0) {
        TimelineOptions.UpdateRealtimePause = TDuration::Seconds(proto.GetUpdateRealtimePause());
    }
}

void NPq2Saas::TDriveTelematicsHandlerBaseSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* BackendEndpoints = " << JoinRange(",", BackendEndpoints.begin(), BackendEndpoints.end()) << Endl;
    stream << "* BackendTagFilter = " << BackendTagFilter << Endl;
    stream << "* FrontendApiHostPort = " << FrontendApiHostPort << Endl;
    stream << "* FrontendApiExtraCgi = " << FrontendApiExtraCgi << Endl;
    stream << "* DestinationName = " << DestinationName << Endl;
    stream << "* Fraction = " << Fraction << Endl;
    stream << "* MaxAge = " << MaxAge << Endl;
    stream << "* ModelsWhitelist = " << JoinRange(",", ModelsWhitelist.begin(), ModelsWhitelist.end()) << Endl;
    stream << "* StatusWhitelist = " << JoinRange(",", StatusWhitelist.begin(), StatusWhitelist.end()) << Endl;
    stream << "* ServiceTagsWhitelist = " << JoinRange(",", ServiceTagsWhitelist.begin(), ServiceTagsWhitelist.end()) << Endl;
    stream << "* TagsWhitelist = " << JoinRange(",", TagsWhitelist.begin(), TagsWhitelist.end()) << Endl;
    stream << "* TagsBlacklist = " << JoinRange(",", TagsBlacklist.begin(), TagsBlacklist.end()) << Endl;
    stream << "* Quota = " << Quota << Endl;
}

void NPq2Saas::TDriveTelematicsHandlerBaseSettings::RegisterUnistatSignalHoles(const TString& deliveryName) const {
    TUnistat::Instance().DrillFloatHole(deliveryName + "-sent", "dmmm", NUnistat::TPriority(50));
}

void NPq2Saas::TDriveTelematicsHandlerBaseSettings::Verify(TDependencyManagerPtr) const {
    PrefetchTimelines(*this);
}

NPq2Saas::TDriveTelematicsStateHandlerSettings::TDriveTelematicsStateHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
    , DestinationName(config.GetDriveTelematicsState().GetDestinationName())
    , IndexState(config.GetDriveTelematicsState().GetIndexState())
{
    const auto& driveTelematicsState = config.GetDriveTelematicsState();
    if (driveTelematicsState.HasFraction()) {
        Fraction = driveTelematicsState.GetFraction();
    }
    for (auto&& imei : driveTelematicsState.GetWhitelist()) {
        Whitelist.insert(imei);
    }
}

void NPq2Saas::TDriveTelematicsStateHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* DestinationName = " << DestinationName << Endl;
    stream << "* Fraction = " << Fraction << Endl;
    stream << "* Whitelist.size = " << Whitelist.size() << Endl;
    stream << "* IndexState = " << IndexState << Endl;
}

NPq2Saas::TDriveEGTSHandlerSettings::TDriveEGTSHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config)
    : TBase(config)
    , SendExtendedStatus(config.GetDriveEGTSEvents().GetSendExtendedStatus())
{    
}

void NPq2Saas::TDriveEGTSHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBase::PrintToStream(stream);
    stream << "* SendExtendedStatus = " << SendExtendedStatus << Endl;
}

NPq2Saas::TBaseHandlerSettings::TFactory::TRegistrator<NPq2Saas::TDriveMobileMetrikaHandlerSettings> DriveMobileMetrikaHandlerSettings(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_MOBILE_METRIKA);
NPq2Saas::TBaseHandlerSettings::TFactory::TRegistrator<NPq2Saas::TDriveTelematicsHandlerSettings> DriveTelematicsHandlerSettings(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_TELEMATICS);
NPq2Saas::TBaseHandlerSettings::TFactory::TRegistrator<NPq2Saas::TDriveTelematicsStateHandlerSettings> DriveTelematicsStateHandlerSettings(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_TELEMATICS_STATE);
NPq2Saas::TBaseHandlerSettings::TFactory::TRegistrator<NPq2Saas::TDriveEGTSHandlerSettings> DriveEGTSHandlerSettings(NPq2SaasProto::THandlerSpecificConfig_EHandlerType_DRIVE_EGTS);
