#pragma once

#include <drive/pq2saas/libcaches/uuid_meta_cache.h>

#include <rtline/api/indexing_client/client.h>
#include <rtline/api/search_client/client.h>

#include <util/generic/ptr.h>
#include <util/generic/string.h>


namespace NPq2Saas {

class IResource {
public:
    virtual ~IResource() {};
};

class IDestination: public IResource {
};

class ICache: public IResource {
};

class TUUIDMetaCacheResource: public ICache {
public:
    template <typename ...TArgs>
    TUUIDMetaCacheResource(TArgs&& ...args)
        : Cache(MakeAtomicShared<TUUIDMetaCache>(std::forward<TArgs>(args)...))
    {}

    TAtomicSharedPtr<TUUIDMetaCache> GetCache() {
        return Cache;
    }

private:
    TAtomicSharedPtr<TUUIDMetaCache> Cache;
};

struct THandlerDependency {
    enum {SaasIndexing, SaasSearch, Cache, HttpRequest, Redis, MongoInfo, Generic} Type;
    TString Name;
};

class TSaasIndexingDestination: public IDestination {
public:
    TSaasIndexingDestination(const TString& server, ui16 port,
                             const TString& indexToken, ui64 maxRetries,
                             bool instantReply);

    ui64 GetMaxRetries() const {
        return MaxRetries;
    }

    bool GetInstantReply() const {
        return InstantReply;
    }

    TAtomicSharedPtr<NRTLine::TIndexingClient> GetIndexingClient() const {
        return IndexingClient;
    }

private:
    const TString Server;
    const ui16 Port;
    const TString Token;
    ui64 MaxRetries;
    bool InstantReply;

    TAtomicSharedPtr<NRTLine::TIndexingClient> IndexingClient;
};

class TSaasSearchDestination: public IDestination {
public:
    TSaasSearchDestination(const TString& server, ui16 port,
                           const TString& serviceName, ui64 maxRetries);

    ui64 GetMaxRetries() const {
        return MaxRetries;
    }

    TAtomicSharedPtr<NRTLine::TSearchClient> GetSearchClient() const {
        return SearchClient;
    }

private:
    const TString Server;
    const ui16 Port;
    const TString ServiceName;
    ui64 MaxRetries;

    TAtomicSharedPtr<NRTLine::TSearchClient> SearchClient;
};

class TGenericDestination: public IDestination {
public:
    TGenericDestination(const TString& host, ui16 port)
        : Host(host)
        , Port(port)
    {
    }

    const TString& GetHost() const {
        return Host;
    }
    ui16 GetPort() const {
        return Port;
    }

private:
    const TString Host;
    const ui16 Port;
};

class THttpDestination: public IDestination {
public:
    THttpDestination(const TString& url, ui64 maxRetries)
        : Url(url)
        , MaxRetries(maxRetries)
    {}

    const TString& GetUrl() const {
        return Url;
    }

    ui64 GetMaxRetries() const {
        return MaxRetries;
    }

private:
    const TString Url;
    ui64 MaxRetries;
};

class TDependencyManager {
public:
    void AddResource(const TString& name, THolder<IResource>&& resource);

    template <typename T>
    T& GetResource(const TString& name) {
        IResource& baseResource = *Resources.at(name);
        T& resource = dynamic_cast<T&>(baseResource);
        return resource;
    }

private:
    THashMap<TString, THolder<IResource>> Resources;
};

using TDependencyManagerPtr = TAtomicSharedPtr<TDependencyManager>;

} // namespace NPq2Saas
