#include "settings.h"

namespace {

using namespace NPq2Saas;
using namespace NPq2SaasProto;

THolder<TBaseHandlerSettings> ConstructHandlerSettings(EHandlerType type,
                                                       const THandlerSpecificConfig& config)
{
    switch (type) {
        case THandlerSpecificConfig_EHandlerType_BS_MOBILE:
            return THolder(new TBsMobileHandlerSettings(config));
        case THandlerSpecificConfig_EHandlerType_BS_MOBILE_FEEDBACK:
            return THolder(new TBsMobileFeedbackHandlerSettings(config));
        case THandlerSpecificConfig_EHandlerType_ANALYZER_LOCATION:
            return THolder(new TAnalyzerLocationHandlerSettings(config));
        case THandlerSpecificConfig_EHandlerType_ANALYZER_TRACK:
            return THolder(new TAnalyzerTrackHandlerSettings(config));
        case THandlerSpecificConfig_EHandlerType_BACKA_COMPANIES:
            return THolder(new TBackaCompaniesHandlerSettings(config));
        case THandlerSpecificConfig_EHandlerType_PUSHITER:
            return THolder(new TPushesDataProcessorHandlerSettings(config));
        case THandlerSpecificConfig_EHandlerType_DEV_NULL:
            return THolder(new TDevNullHandlerSettings(config));
        case THandlerSpecificConfig_EHandlerType_PUSHER_PROPS:
            return THolder(new TPusherPropsHandlerSettings(config));
        default:
            return THolder(TBaseHandlerSettings::TFactory::Construct(type, config));
    }
}
} // anonymous namespace

namespace NPq2Saas {

using namespace NPq2SaasProto;

const TString& GetHandlerTypeName(EHandlerType type) {
    return THandlerSpecificConfig_EHandlerType_Name(type);
}

TFilteringSettings::TFilteringSettings(const NPq2SaasProto::TFilteringConfig& config)
    : EventInitialLagThreshold(TDuration::Minutes(config.GetEventInitialLagThreshold()))
    , AcceptedAppIds(config.GetAcceptedAppIds().begin(), config.GetAcceptedAppIds().end())
    , AcceptedEventNames(config.GetAcceptedEventNames().begin(), config.GetAcceptedEventNames().end())
{}

void TFilteringSettings::PrintToStream(IOutputStream& stream) const {
    stream << "\n[FILTERING SETTINGS]:\n";
    stream << "* LagFilter = " << EventInitialLagThreshold << Endl;
    stream << "* AppIds [" << AcceptedAppIds.size() << "]:\n";
    ForEach(AcceptedAppIds.begin(), AcceptedAppIds.end(),
            [&stream](const TString& s) { stream << "  " << s << '\n'; });
    stream << "* AppEventNames [" << AcceptedEventNames.size() << "]:\n";
    ForEach(AcceptedEventNames.begin(), AcceptedEventNames.end(),
            [&stream](const TString& s) { stream << "  " << s << '\n'; });
}

const THashSet<TString>& TFilteringSettings::GetAcceptedAcceptedAppIds() const {
    return AcceptedAppIds;
}

const THashSet<TString>& TFilteringSettings::GetAcceptedEventNames() const {
    return AcceptedEventNames;
}

bool TFilteringSettings::IsNeededAppId(const TString& appId) const {
    return AcceptedAppIds.contains(appId);
}

bool TFilteringSettings::IsNeededEventName(const TString& eventName) const {
    return AcceptedEventNames.contains(eventName);
}

TSpyByUUIDSettings::TSpyByUUIDSettings(const NPq2SaasProto::TSpyByUUIDConfig& config)
    : SpyByFullUUID(config.GetFullMatches().begin(), config.GetFullMatches().end())
    , SpyByUUIDPrefix(config.GetPrefixMatches().begin(), config.GetPrefixMatches().end())
{}

void TSpyByUUIDSettings::PrintToStream(IOutputStream& stream) const {
    stream << "\n[SPY SETTINGS]:\n";
    stream << "* FullUUID [" << SpyByFullUUID.size() << "]:\n";
    ForEach(SpyByFullUUID.begin(), SpyByFullUUID.end(),
            [&stream](const TString& s) { stream << "  " << s << '\n'; });
    stream << "* UUIDPrefix [" << SpyByUUIDPrefix.size() << "]:\n";
    ForEach(SpyByUUIDPrefix.begin(), SpyByUUIDPrefix.end(),
            [&stream](const TString& s) { stream << "  " << s << '\n'; });
}

bool TSpyByUUIDSettings::ShouldSpyOnUUID(const TString& uuid) const {
    return SpyByFullUUID.contains(uuid) ||
           AnyOf(SpyByUUIDPrefix,
                 [&uuid] (const TString& prefix) { return uuid.StartsWith(prefix); });
}

TBaseHandlerSettings::TBaseHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config)
    : HandlerType(config.GetHandlerType())
    , Filtering(config.GetCommon().GetFiltering())
    , Spy(config.GetCommon().GetSpyByUUID())
{}

THandlerSettings::THandlerSettings(const THandlerSpecificConfig& config)
    : Settings(ConstructHandlerSettings(config.GetHandlerType(), config))
{}

TBsMobileHandlerSettings::TBsMobileHandlerSettings(const THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
    , SaasIndexingDestinationName(config.GetBsMobile().GetDestinationName())
    , SaasSearchUserKVName(config.GetBsMobile().GetReadUserMetaKVName())
    , SaasIndexingUserKVName(config.GetBsMobile().GetWriteUserMetaKVName())
    , GeoShardingEnabled(config.GetBsMobile().GetGeoShardingEnabled())
{}

TBsMobileFeedbackHandlerSettings::TBsMobileFeedbackHandlerSettings(const THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
    , HttpRequestDestinationName("feedback_http_backend")
    , SaasFeedbackBackends(config.GetBsMobileFeedback().GetFeedbackBackends().begin(),
                           config.GetBsMobileFeedback().GetFeedbackBackends().end())
{}

TAnalyzerLocationHandlerSettings::TAnalyzerLocationHandlerSettings(const THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
    , SaasIndexingDestinationName(config.GetAnalyzerLocation().GetDestinationName()) {
}

TAnalyzerTrackHandlerSettings::TAnalyzerTrackHandlerSettings(const THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
    , SaasIndexingDestinationName(config.GetAnalyzerTrack().GetDestinationName())
    , CheckUserAcceptance(config.GetAnalyzerTrack().GetCheckUserAcceptance())
    , SaasSearchUserKVName(config.GetAnalyzerTrack().GetReadUserMetaKVName())
    , UUIDMetaCacheName(config.GetAnalyzerTrack().GetUUIDMetaCacheName())
    , GeoShardingEnabled(config.GetAnalyzerTrack().GetGeoShardingEnabled())
    , IgnoredClids(config.GetAnalyzerTrack().GetIgnoredClids().begin(), config.GetAnalyzerTrack().GetIgnoredClids().end())
    , Fraction(config.GetAnalyzerTrack().GetFraction())
{
    // Temporary ignore artificial Paris tracks.
    IgnoredClids.insert("J421");
    IgnoredClids.insert("J422");
}

TBackaCompaniesHandlerSettings::TBackaCompaniesHandlerSettings(const THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
    , HttpRequestDestinationName(config.GetBackaCompanies().GetDestinationName())
{}

TPushesDataProcessorHandlerSettings::TPushesDataProcessorHandlerSettings(
    const NPq2SaasProto::THandlerSpecificConfig &config)
    : TBaseHandlerSettings(config)
    , RedisDestinationName(config.GetPushiter().GetRedisDestinationName())
    , MongoInfoName(config.GetPushiter().GetMongoInfoName())
{}

TDevNullHandlerSettings::TDevNullHandlerSettings(const THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
{}

TPusherPropsHandlerSettings::TPusherPropsHandlerSettings(const THandlerSpecificConfig& config)
    : TBaseHandlerSettings(config)
    , SaasIndexingDestinationName(config.GetPusherProps().GetDestinationName())
    , UrlFields(config.GetPusherProps().GetUrlFields().begin(), config.GetPusherProps().GetUrlFields().end())
    , UrlPrefix(config.GetPusherProps().GetUrlPrefix())
    , IgnoringFields(config.GetPusherProps().GetIgnoringFields().begin(), config.GetPusherProps().GetIgnoringFields().end())
{
    ui64 onlyFieldsSize = config.GetPusherProps().OnlyFieldsSize();
    if (onlyFieldsSize) {
        ui64 newNamesSize = config.GetPusherProps().ResultFieldsSize();

        if (!newNamesSize && newNamesSize != onlyFieldsSize) {
            ythrow yexception() << "invalid PusherProps.ResultFields count." ;
        }

        for (ui32 i = 0; i < onlyFieldsSize; ++i) {
            const TString oldName = config.GetPusherProps().GetOnlyFields(i);
            const TString newName = newNamesSize ? config.GetPusherProps().GetResultFields(i) : oldName;
            OnlyFields[oldName] = newName;
        }
    }

    for (const auto& filter : config.GetPusherProps().GetFilters()) {
        if (!filter.GetField() || !filter.GetRegExp()) {
            ythrow yexception() << "invalid PusherProps.Filters." << filter.GetField() << ":" << filter.GetRegExp();
        }
        Filters.emplace(filter.GetField(), filter.GetRegExp());
    }
}

void TBsMobileHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* SaasIndexingDestinationName = " << SaasIndexingDestinationName << Endl;
    stream << "* SaasSearchUserKVName = " << SaasSearchUserKVName << Endl;
    stream << "* SaasIndexingUserKVName = " << SaasIndexingUserKVName << Endl;
    stream << "* GeoShardingEnabled = " << GeoShardingEnabled << Endl;
}

void TBsMobileFeedbackHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* FeedbackBackends:" << Endl;
    for (const auto& backend : SaasFeedbackBackends) {
        stream << " - " << backend << Endl;
    }
}

void TAnalyzerLocationHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* SaasIndexingDestinationName = " << SaasIndexingDestinationName << Endl;
}

void TAnalyzerTrackHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* SaasIndexingDestinationName = " << SaasIndexingDestinationName << Endl;
    stream << "* CheckUserAcceptance = " << CheckUserAcceptance << Endl;
    if (CheckUserAcceptance) {
        stream << "* SaasSearchUserKVName = " << SaasSearchUserKVName << Endl;
        stream << "* UUIDMetaCacheName = " << UUIDMetaCacheName << Endl;
    }
    stream << "* GeoShardingEnabled = " << GeoShardingEnabled << Endl;
}

void TBackaCompaniesHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* HttpRequestDestinationName = " << HttpRequestDestinationName << Endl;
}

void TPushesDataProcessorHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* RedisDestinationName = " << RedisDestinationName << Endl;
    stream << "* MongoInfoName = " << MongoInfoName << Endl;
}

void TDevNullHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
}

void TPusherPropsHandlerSettings::PrintToStream(IOutputStream& stream) const {
    TBaseHandlerSettings::PrintToStream(stream);
    stream << "* SaasIndexingDestinationName = " << SaasIndexingDestinationName << Endl;
    stream << "* UrlFileds = [";
    for (const auto& field : UrlFields) {
        stream << field << ",";
    }
    stream << "]" << Endl;
    stream << "* UrlPrefix = " << UrlPrefix << Endl;
    stream << "* IgnoringFields = [";
    for (const auto& field : IgnoringFields) {
        stream << field << ",";
    }
    stream << "]" << Endl;
}

} // namespace NPq2Saas
