#pragma once

#include "dependencies.h"

#include <drive/pq2saas/protos/handler_config.pb.h>
#include <drive/pq2saas/protos/pq2saas_config.pb.h>

#include <library/cpp/object_factory/object_factory.h>
#include <library/cpp/regex/pcre/regexp.h>
#include <util/datetime/base.h>
#include <util/generic/hash_set.h>
#include <util/generic/maybe.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/stream/output.h>
#include <util/string/cast.h>


namespace NPq2Saas {

using EHandlerType = NPq2SaasProto::THandlerSpecificConfig::EHandlerType;

const TString& GetHandlerTypeName(EHandlerType handlerType);

class IPrintable {
public:
    virtual ~IPrintable() {};
    virtual void PrintToStream(IOutputStream& /* stream */) const {};
};

class TFilteringSettings: public IPrintable {
public:
    TFilteringSettings(const NPq2SaasProto::TFilteringConfig& config);

    void PrintToStream(IOutputStream& stream) const override;

    const THashSet<TString>& GetAcceptedAcceptedAppIds() const;
    const THashSet<TString>& GetAcceptedEventNames() const;
    bool IsNeededAppId(const TString& appId) const;
    bool IsNeededEventName(const TString& eventName) const;
    bool IsNotTooOld(TDuration ) const;

public:
    const TDuration EventInitialLagThreshold;

private:
    THashSet<TString> AcceptedAppIds;
    THashSet<TString> AcceptedEventNames;
};

class TSpyByUUIDSettings: public IPrintable {
public:
    TSpyByUUIDSettings(const NPq2SaasProto::TSpyByUUIDConfig& config);

    void PrintToStream(IOutputStream& stream) const override;

    bool ShouldSpyOnUUID(const TString& uuid) const;

private:
    THashSet<TString> SpyByFullUUID;
    TVector<TString> SpyByUUIDPrefix;
};

class TBaseHandlerSettings: public IPrintable {
public:
    TBaseHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    virtual void Verify(TDependencyManagerPtr) const {}
    virtual TVector<THandlerDependency> GetDependencies() const { return {}; }
    void PrintToStream(IOutputStream& stream) const override {
        stream << "HandlerType: " << GetHandlerTypeName(HandlerType) << Endl;
        Filtering.PrintToStream(stream);
        Spy.PrintToStream(stream);
    }
    virtual void RegisterUnistatSignalHoles(const TString& /*deliveryName*/) const {};

    using TFactory = NObjectFactory::TParametrizedObjectFactory<TBaseHandlerSettings, EHandlerType, const NPq2SaasProto::THandlerSpecificConfig&>;

public:
    EHandlerType HandlerType;
    TFilteringSettings Filtering;
    TSpyByUUIDSettings Spy;
};

class TBsMobileHandlerSettings: public TBaseHandlerSettings {
public:
    TBsMobileHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    void PrintToStream(IOutputStream& stream) const override;
    TVector<THandlerDependency> GetDependencies() const override {
        return {
            {THandlerDependency::SaasIndexing, SaasIndexingDestinationName},
            {THandlerDependency::SaasSearch, SaasSearchUserKVName},
            {THandlerDependency::SaasIndexing, SaasIndexingUserKVName},
        };
    }

public:
    TString SaasIndexingDestinationName;
    TString SaasSearchUserKVName;
    TString SaasIndexingUserKVName;
    bool GeoShardingEnabled;
};

class TBsMobileFeedbackHandlerSettings: public TBaseHandlerSettings {
public:
    TBsMobileFeedbackHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    void PrintToStream(IOutputStream& stream) const override;
    void Verify(TDependencyManagerPtr) const override {
        Y_VERIFY(SaasFeedbackBackends.size() > 0, "There should be more than zero feedback backends");
    }
    TVector<THandlerDependency> GetDependencies() const override {
        return {
            {THandlerDependency::HttpRequest, HttpRequestDestinationName},
        };
    }

public:
    // FIXME: For now this variable is a dummy - only for autocreation of deliveryDestinationStats
    TString HttpRequestDestinationName;
    TVector<TString> SaasFeedbackBackends;
};

class TAnalyzerLocationHandlerSettings: public TBaseHandlerSettings {
public:
    TAnalyzerLocationHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    void PrintToStream(IOutputStream& stream) const override;
    TVector<THandlerDependency> GetDependencies() const override {
        return{
            { THandlerDependency::SaasIndexing, SaasIndexingDestinationName },
        };
    }

public:
    TString SaasIndexingDestinationName;
};

struct TAnalyzerTrackHandlerSettings: public TBaseHandlerSettings {
public:
    TAnalyzerTrackHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    void PrintToStream(IOutputStream& stream) const override;
    TVector<THandlerDependency> GetDependencies() const override {
        if (CheckUserAcceptance) {
            return {
                {THandlerDependency::SaasIndexing, SaasIndexingDestinationName},
                {THandlerDependency::SaasSearch, SaasSearchUserKVName},
                {THandlerDependency::Cache, UUIDMetaCacheName},
            };
        } else {
            return {
                {THandlerDependency::SaasIndexing, SaasIndexingDestinationName},
            };
        }
    }

public:
    TString SaasIndexingDestinationName;
    bool CheckUserAcceptance;
    TString SaasSearchUserKVName;
    TString UUIDMetaCacheName;
    bool GeoShardingEnabled;
    THashSet<TString> IgnoredClids;
    float Fraction;
};

class TBackaCompaniesHandlerSettings: public TBaseHandlerSettings {
public:
    TBackaCompaniesHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    void PrintToStream(IOutputStream& stream) const override;
    TVector<THandlerDependency> GetDependencies() const override {
        return {
            {THandlerDependency::HttpRequest, HttpRequestDestinationName},
        };
    }

public:
    TString HttpRequestDestinationName;
};


class TPushesDataProcessorHandlerSettings: public TBaseHandlerSettings {
public:
    TPushesDataProcessorHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    void PrintToStream(IOutputStream& stream) const override;
    TVector<THandlerDependency> GetDependencies() const override {
        return {
           {THandlerDependency::Redis, RedisDestinationName},
           {THandlerDependency::MongoInfo, MongoInfoName}
        };
    }

public:
    TString RedisDestinationName;
    TString MongoInfoName;
};

class TDevNullHandlerSettings: public TBaseHandlerSettings {
public:
    TDevNullHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    void PrintToStream(IOutputStream& stream) const override;
    TVector<THandlerDependency> GetDependencies() const override {
        return {};
    }
};

struct TPusherPropsHandlerSettings : public TBaseHandlerSettings {
public:
    TPusherPropsHandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    void PrintToStream(IOutputStream& stream) const override;
    TVector<THandlerDependency> GetDependencies() const override {
        return {
            { THandlerDependency::SaasIndexing, SaasIndexingDestinationName }
        };
    }

public:
    TString SaasIndexingDestinationName;
    TVector<TString> UrlFields;
    TString UrlPrefix;
    THashSet<TString> IgnoringFields;
    THashMap<TString, TString> OnlyFields;
    THashMap<TString, TRegExMatch> Filters;
};

class THandlerSettings {
public:
    THandlerSettings(const NPq2SaasProto::THandlerSpecificConfig& config);

    void Verify(TDependencyManagerPtr manager) const {
        Settings->Verify(manager);
    }

    void PrintToStream(IOutputStream& stream) const {
        Settings->PrintToStream(stream);
    }

    template <typename T>
    const T& Get() const {
        return dynamic_cast<const T&>(*Settings);
    }
    const TBaseHandlerSettings* GetInterfacePtr() const {
        return Settings.Get();
    }

private:
    TAtomicSharedPtr<TBaseHandlerSettings> Settings;
};

} // namespace NPq2Saas
