#include "manager.h"

#include <util/datetime/base.h>
#include <util/generic/vector.h>
#include <util/string/cast.h>


namespace {

TVector<TDuration> GenerateSecondsTimeBuckets() {
    TVector<TDuration> secondsBuckets;
    for (ui64 i = 1; i < 5; ++i) { // 1, 2, .. 4 sec
        secondsBuckets.emplace_back(TDuration::Seconds(i));
    }
    for (ui64 i = 5; i < 60; i += 5) { // 5, 10, .. 55 sec
        secondsBuckets.emplace_back(TDuration::Seconds(i));
    }
    for (ui64 i = 1; i < 10; ++i) { // 1, 2, .. 9 min
        secondsBuckets.emplace_back(TDuration::Minutes(i));
    }
    for (ui64 i = 10; i < 60; i += 10) { // 10, 20, .. 50 min
        secondsBuckets.emplace_back(TDuration::Minutes(i));
    }
    for (ui64 i = 1; i < 6; ++i) { // 1, 2, .. 5 h
        secondsBuckets.emplace_back(TDuration::Hours(i));
    }
    for (ui64 i = 6; i <= 24; i += 6) { // 6, 12, 18, 24 h
        secondsBuckets.emplace_back(TDuration::Hours(i));
    }
    return secondsBuckets;
}

TVector<TDuration> GenerateMillisecondsTimeBuckets() {
    TVector<TDuration> msBuckets;
    for (ui64 i = 1; i < 5; ++i) { // 1, 2, .. 4 ms
        msBuckets.emplace_back(TDuration::MilliSeconds(i));
    }
    for (ui64 i = 5; i < 50; i += 5) { // 5, 10, .. 45 ms
        msBuckets.emplace_back(TDuration::MilliSeconds(i));
    }
    for (ui64 i = 50; i < 100; i += 10) { // 50, 60, .. 90 ms
        msBuckets.emplace_back(TDuration::MilliSeconds(i));
    }
    for (ui64 i = 100; i < 1000; i += 50) { // 100, 150, .. 950 ms
        msBuckets.emplace_back(TDuration::MilliSeconds(i));
    }
    for (ui64 i = 1; i < 6; ++i) { // 1, 2, .. 5 s
        msBuckets.emplace_back(TDuration::Seconds(i));
    }
    return msBuckets;
}

const TVector<TDuration> SEC_BUCKETS = GenerateSecondsTimeBuckets();
const TVector<TDuration> MS_BUCKETS = GenerateMillisecondsTimeBuckets();

void GenerateNumericCounters(int min, int max, NPq2SaasMonitoring::TCountersPtr countersGroup,
                             TVector<NPq2SaasMonitoring::TCounterPtr>& result)
{
    Y_VERIFY(min <= max, "Min should be less or equal to max");
    result.clear();
    result.reserve(max - min + 1);
    for (int i = min; i <= max; ++i) {
        result.push_back(countersGroup->GetCounter(ToString(i + 1), true));
    }
}

const size_t BLOB_BUCKET_SIZE_KB = 512;
const size_t BLOB_BUCKET_COUNT = 20;

} // anonymous namespace

namespace NPq2SaasMonitoring {

TManager::TManager(const THashSet<TString>& acceptedEventNames)
    : Counters(GetMonitoring().GetCounters())
    , LockedTopics(Counters->GetCounter("LockedTopicsCount"))
    , ExclusiveReadLockedTopics(Counters->GetCounter("ExclusiveReadLockedTopicsCount"))
    , BlockedConsumerThreads(Counters->GetCounter("BlockedConsumerThreadsCount"))
    , UnsupportedLogTypeMessages(Counters->GetCounter("UnsupportedLogTypeMessages"))
    , LogFormatErrors(Counters->GetCounter("LogFormatErrors", true))
    , EventsFromFuture(Counters->GetCounter("EventsFromFuture", true))
    , ProtoParsingErrors(Counters->GetCounter("ProtoParsingErrorsCount", true))
    , JsonParsingErrors(Counters->GetCounter("JsonParsingErrorsCount", true))
    , JsonDocumentFormatErrors(Counters->GetCounter("JsonDocumentFormatErrorsCount", true))
    , MatchingProtos(Counters->GetCounter("MatchingProtosCount", true))
    , NonMatchingProtos(Counters->GetCounter("NonMatchingProtosCount", true))
    , EventsWithNeededAppId(Counters->GetCounter("EventsWithNeededAppId", true))
    , EventsOfNeededType(Counters->GetCounter("EventsOfNeededType", true))
    , EventsWithInvalidPosition(Counters->GetCounter("EventsWithInvalidPosition", true))
    , EventsWithAccountId(Counters->GetCounter("EventsOfNeededTypeWithAccountId", true))
    , EventsWithoutAccountId(Counters->GetCounter("EventsOfNeededTypeWithoutAccountId", true))
    , EventInitialLag(Counters->GetSubgroup("timeDistribution", "eventInitialLag"),
                      NTimeStats::ETimeAccuracy::SECONDS)

{
    EventInitialLag.AddBuckets(SEC_BUCKETS);
    for (const auto& name : acceptedEventNames) {
        EventName2Counter.insert({
            name, Counters->GetSubgroup("eventName", name)->GetCounter("EventsCount", true)
        });
    }
    BlobSizeDistribution.reserve(BLOB_BUCKET_COUNT);
    auto blobSizeRoot = Counters->GetSubgroup("numericHistogram", "blobSize");
    for (size_t blobBucketIdx = 0; blobBucketIdx < BLOB_BUCKET_COUNT; ++blobBucketIdx) {
        BlobSizeDistribution.push_back(
            blobSizeRoot->GetCounter(ToString((blobBucketIdx + 1) * BLOB_BUCKET_SIZE_KB) + "Kb", true)
        );
    }
}

void TManager::AddHandlerFailCounter(const TString& handler) {
    HandlerFailCounters.insert({
        handler, Counters->GetSubgroup("handlerName", handler)->GetCounter("HandlerFails", true)
    });
}

TCounterPtr TManager::GetHandlerFailCounter(const TString& handler) const {
    return HandlerFailCounters.at(handler);
}

void TManager::AddBadHandlerInputCounter(const TString& handler) {
    BadHandlerInputCounters.insert({
        handler, Counters->GetSubgroup("handlerName", handler)->GetCounter("BadHandlerInputs", true)
    });
}

TCounterPtr TManager::GetBadHandlerInputCounter(const TString& handler) const {
    return BadHandlerInputCounters.at(handler);
}

void TManager::ReportBlobSize(size_t size) {
    size_t bucketIdx = std::min(size / (BLOB_BUCKET_SIZE_KB * 1024),
                                BLOB_BUCKET_COUNT - 1);
    BlobSizeDistribution[bucketIdx]->Inc();
}

TManager::TDeliveryStatsPtr TManager::AddDeliveryStats(const TString& name) {
    auto result = MakeAtomicShared<TDeliveryStats>(Counters->GetSubgroup("delivery", name));
    DeliveryStats.insert({name, result});
    return result;
}

TManager::TDeliveryStatsPtr TManager::GetDeliveryStats(const TString& deliveryName) const {
    return DeliveryStats.at(deliveryName);
}

TManager::TDeliveryDependencyStatsPtr TManager::TDeliveryStats::AddPerDependencyStats(const TString& name) {
    auto result = MakeAtomicShared<TDeliveryDependencyStats>(Root->GetSubgroup("destination", name));
    Stats.insert({name, result});
    return result;
}

TManager::TDeliveryStats::TDeliveryStats(TCountersPtr root)
    : Root(root)
    , TooOldEventsCount(root->GetCounter("TooOldEventsCount", true))
    , EventsWithForbiddenAccountsCount(root->GetCounter("EventsWithForbiddenAccountsCount", true))
    , EventsWithIgnoredClidsCount(root->GetCounter("EventsWithIgnoredClidsCount", true))
    , SaasEventTs(root->GetSubgroup("timeDistribution", "saasEventTs"),
                  NTimeStats::ETimeAccuracy::SECONDS)
    , MetrikaEventLogBrokerTime(root->GetSubgroup("timeDistribution", "metrikaEventLogBrokerTime"),
                                NTimeStats::ETimeAccuracy::SECONDS)
    , MetrikaEventSendTime(root->GetSubgroup("timeDistribution", "metrikaEventSendTime"),
                           NTimeStats::ETimeAccuracy::SECONDS)
    , EventProcessingTime(root->GetSubgroup("timeDistribution", "eventProcessingTime"),
                          NTimeStats::ETimeAccuracy::MILLISECONDS)
{
    SaasEventTs.AddBuckets(SEC_BUCKETS);
    MetrikaEventLogBrokerTime.AddBuckets(SEC_BUCKETS);
    MetrikaEventSendTime.AddBuckets(SEC_BUCKETS);
    EventProcessingTime.AddBuckets(MS_BUCKETS);
}

TManager::TDeliveryDependencyStats::TDeliveryDependencyStats(TCountersPtr root)
    : Root(root)
    , EventsSentFailedNonRetriable(root->GetCounter("EventsSentFailedNonRetriable", true))
    , EventsSentFailedAndRetried(root->GetCounter("EventsSentFailedAndRetried", true))
    , EventsSentFailedMaxRetries(root->GetCounter("EventsSentFailedMaxRetries", true))
    , EventsSentSucceded(root->GetCounter("EventsSentSucceded", true))
    , EventsSentTotal(root->GetCounter("EventsSentTotal", true))
    , EventPreSaasLifetimeCounters(root->GetSubgroup("timeDistribution", "eventPreSaasLifetime"),
                                   NTimeStats::ETimeAccuracy::SECONDS)
    , EventLatencyCounters(root->GetSubgroup("timeDistribution", "eventLatency"),
                           NTimeStats::ETimeAccuracy::SECONDS)
    , SaasReqProcessingTimeCounters(root->GetSubgroup("timeDistribution", "saasRequestProcessingTime"),
                                    NTimeStats::ETimeAccuracy::MILLISECONDS)
    , SaasFailedReqProcessingTimeCounters(root->GetSubgroup("timeDistribution", "saasFailedRequestProcessingTime"),
                                          NTimeStats::ETimeAccuracy::MILLISECONDS)
{
    SaasReqProcessingTimeCounters.AddBuckets(MS_BUCKETS);
    SaasFailedReqProcessingTimeCounters.AddBuckets(MS_BUCKETS);
    EventLatencyCounters.AddBuckets(SEC_BUCKETS);
    EventPreSaasLifetimeCounters.AddBuckets(SEC_BUCKETS);

    // Generate 10 bins.
    // It's assumed that there would be no more than 10 retries.
    // TODO: generate bins on demand.
    GenerateNumericCounters(0, 10,
                            Root->GetSubgroup("numericHistogram", "requestsPerEvent"),
                            ReqPerEventCounters);

}

void TManager::TDeliveryDependencyStats::ReportSaasSendResult(i32 httpCode) {
    Root->GetSubgroup("saasHttpCode", ToString(httpCode))
        ->GetCounter("SaasResponsesCount", true)
        ->Inc();
}

} // namespace NPq2SaasMonitoring
