#include "models.h"
#include "cmath"

#include <util/string/vector.h>
#include <util/string/printf.h>

//TODO recheck all work with mongo

namespace {
//in meters
const ui32 AVG_EARTH_RADIUS = 6371000;
// 500*500 - 24 regions
// 250*250 - 6 regions
const double GRID_STEP = 0.01;

NJson::TJsonValue ParseJsonString(const TString& js) {
    NJson::TJsonValue json;
    TStringStream stream(js);
    const bool readResult = NJson::ReadJsonTree(&stream, true, &json);
    if (!readResult)
        ythrow yexception() << "Failed to parse json";
    return json;
};

template<typename T>
void AddSelectionParam(NJson::TJsonValue& query, const TString& paramName, const TString& selector, T value) {
    NJson::TJsonValue p(NJson::JSON_MAP);
    p[selector] = value;
    query[paramName] = p;
}

double DegreeToRadian(double deg) {
    return deg * M_PI / 180.0;
}

double PointsDistance(double lat1, double lon1, double lat2, double lon2) {
    lat1 = DegreeToRadian(lat1);
    lon1 = DegreeToRadian(lon1);
    lat2 = DegreeToRadian(lat2);
    lon2 = DegreeToRadian(lon2);
    double lat = lat2 - lat1;
    double lon = lon2 - lon1;
    double d = pow(sin(lat * 0.5), 2) + cos(lat1) * cos(lat2) * pow(sin(lon * 0.5), 2);
    double h = 2 * AVG_EARTH_RADIUS * asin(sqrt(d));
    return h / 1000;
}


}// anonymous namespace


namespace NPushiter {

HashType GenerateLocationHash(double lat, double lon) {
    return ((i64) (lat / GRID_STEP)) * 10 * (ui64)(1 / GRID_STEP) + (i64) (lon / GRID_STEP);
}

TPoint::TPoint(double lat, double lon)
    : Lat(lat)
    , Lon(lon) {}

double TPoint::DistanceTo(const TPoint& p) const {
    return PointsDistance(this->Lat, this->Lon, p.Lat, p.Lon);
}

TPoint TPoint::FromJsonMap(const THashMap<TString, NJson::TJsonValue>& map) {
    double lat = map.at("lat").GetDouble();
    double lon = map.at("lon").GetDouble();
    return TPoint(lat, lon);
}

double TCircle::DistanceTo(const TPoint& p) const {
    return Center.DistanceTo(p);
}

bool TCircle::IsInside(const TPoint& p) const {
    return DistanceTo(p) < Radius;
}

TVector<HashType> TCircle::GenerateGrid() const {
    TVector<HashType> grid;
    const double offsetLat = Radius / AVG_EARTH_RADIUS;
    const double offsetLon = Radius / (AVG_EARTH_RADIUS * cos(M_PI * Center.GetLat() / 180));
    for (double curLat = Center.GetLat() - offsetLat; curLat <= Center.GetLat() + offsetLat; curLat += GRID_STEP) {
        for (double curLon = Center.GetLon() - offsetLon; curLon <= Center.GetLon() + offsetLon; curLon += GRID_STEP) {
            grid.push_back(GenerateLocationHash(curLat, curLon));
        }
    }
    return grid;

}

TCircle TCircle::FromJson(const NJson::TJsonValue& js) {
    THashMap<TString, NJson::TJsonValue> map = js.GetMap();
    double radius = map["radius"].GetDouble() / 1000;
    TPoint center = TPoint::FromJsonMap(map["center"].GetMap());
    return TCircle(radius, center);
}


TPushiterDbObject TPushiterDbObject::FromJson(const NJson::TJsonValue& js) {
    auto map = js.GetMap();
    auto idWrapper = map["_id"].GetMap();
    TString id = idWrapper["$oid"].GetString();
    TString topic = map["topic"].GetString();
    const TString geometryString = map["geometry"].GetString();
    TCircle geometry = TCircle::FromJson(ParseJsonString(geometryString));
    return TPushiterDbObject(id, topic, geometry);
}

TPushiterDbObject::TPushiterDbObject(const TString& id, const TString& topic,
                                     const TCircle& geometry)
    : Id(id)
    , Topic(topic)
    , Geometry(geometry) {}

bool TPushiterDbObject::IsInside(const TPoint& location) const {
    return Geometry.IsInside(location);
}

TString TPushiterDbObject::GetId() const {
    return Id;
}

TString TPushiterDbObject::GetTopic() const {
    return Topic;
}

TVector<HashType> TPushiterDbObject::GenerateGrid() const {
    return Geometry.GenerateGrid();
}

}
