#include <drive/pq2saas/libcaches/uuid_meta_cache.h>
#include <drive/pq2saas/libconfig/config.h>
#include <drive/pq2saas/libconsumer/consumer.h>
#include <drive/pq2saas/libconsumer/settings.h>
#include <drive/pq2saas/liblogging/log.h>
#include <drive/pq2saas/libmonitoring/monitoring.h>
#include <drive/pq2saas/libprobes/consumer.h>
#include <drive/pq2saas/libprobes/register.h>
#include <drive/pq2saas/libpushiter/resources.h>

#include <drive/library/cpp/threading/future.h>

#include <google/protobuf/text_format.h>

#include <kikimr/persqueue/sdk/deprecated/cpp/v2/persqueue.h>
#include <kikimr/persqueue/sdk/deprecated/cpp/v1/consumer.h>
#include <kikimr/persqueue/sdk/deprecated/cpp/v1/persqueue.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/sighandler/async_signals_handler.h>
#include <library/cpp/threading/future/future.h>

#include <rtline/library/storage/abstract.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/datetime/base.h>
#include <util/generic/cast.h>
#include <util/generic/string.h>
#include <util/stream/file.h>
#include <util/stream/output.h>
#include <util/stream/str.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <util/system/env.h>
#include <util/system/event.h>
#include <util/system/hostname.h>
#include <util/system/yassert.h>

#include <functional>
#include <utility>


namespace {

struct TFnProxy {
    std::function<void()> Fn;
};

class TMonitoringGuard {
public:
    TMonitoringGuard(NPq2SaasMonitoring::TPq2SaasMonitoring& monitoring)
        : Monitoring(monitoring)
    {
        INFO_LOG << "Starting monitoring" << Endl;
        Monitoring.Start();
        INFO_LOG << "Started monitoring" << Endl;
    }
    ~TMonitoringGuard() {
        INFO_LOG << "Shutting down monitoring" << Endl;
        Monitoring.Shutdown();
        INFO_LOG << "Shut down monitoring" << Endl;
    }

private:
    NPq2SaasMonitoring::TPq2SaasMonitoring& Monitoring;
};

class TConsumerCallbackManagerGuard {
public:
    TConsumerCallbackManagerGuard(NPq2Saas::TConsumerCallbackManager& manager)
        : Manager(manager)
    {
    }
    ~TConsumerCallbackManagerGuard() {
        INFO_LOG << "Stopping ConsumerCallbackManager" << Endl;
        Manager.RequestStop();
        Manager.Wait();
        INFO_LOG << "Stopped ConsumerCallbackManager" << Endl;
    }

private:
    NPq2Saas::TConsumerCallbackManager& Manager;
};

void Log(const TString& msg) {
    Cerr << '[' << Now().ToString() << "] " << msg << Endl;
}

TString ConvertLocationToDC(const TString& location) {
    Y_VERIFY(EqualToOneOf(location, "msk", "man", "sas", "vla"), "location should be one of {msk, man, sas, vla}");
    if (EqualToOneOf(location, "msk")) {
        // TODO: Remove after LB is deployed in Vladimir LBOPS-803.
        return "myt";
    }
    return location;
}

/**
 * @brief Get name of current datacenter based on location
 *
 * @returns One of {"man", "myt", "sas", "vla"}
 */
TString GetCurrentDC() {
    if (const TString location = GetEnv("DC_GEO")) {
        return ConvertLocationToDC(location);
    }

    for (const TString dc : {"sas", "man", "vla"}) {
        if (HostName().StartsWith(dc)) {
            return dc;
        }
    }
    return "myt";
}

THolder<NPq2Saas::IResource> ParseDestinationConfig(NPq2SaasProto::TDestinationConfig config) {
    switch (config.GetDestinationType()) {
        case NPq2SaasProto::TDestinationConfig::GENERIC:
            return MakeHolder<NPq2Saas::TGenericDestination>(config.GetServer(), IntegerCast<ui16>(config.GetPort()));
        case NPq2SaasProto::TDestinationConfig::SEARCH:
            return MakeHolder<NPq2Saas::TSaasSearchDestination>(
                config.GetServer(), static_cast<ui16>(config.GetPort()),
                config.GetServiceName(), config.GetMaxRetries()
            );
        case NPq2SaasProto::TDestinationConfig::INDEX:
            return MakeHolder<NPq2Saas::TSaasIndexingDestination>(
                config.GetServer(), static_cast<ui16>(config.GetPort()),
                config.GetToken(), config.GetMaxRetries(), config.GetInstantReply()
            );
        case NPq2SaasProto::TDestinationConfig::HTTP:
            return MakeHolder<NPq2Saas::THttpDestination>(config.GetUrl(), config.GetMaxRetries());
        case NPq2SaasProto::TDestinationConfig::REDIS:
            return MakeHolder<NPq2Saas::TRedisDestination>(config.GetServer(), config.GetPort(), config.GetPoolSize());
        case NPq2SaasProto::TDestinationConfig::PUSHITER_MONGO:
            ythrow yexception() << "deprecated";
    }
}

THolder<NPq2Saas::IResource> ParseCacheConfig(NPq2SaasProto::TTimedCacheConfig config) {
    return MakeHolder<NPq2Saas::TUUIDMetaCacheResource>(
        TDuration::MilliSeconds(config.GetTimeTillOutdatedMs()),
        TDuration::MilliSeconds(config.GetTimeTillTryUpdateMs()),
        TDuration::MilliSeconds(config.GetUpdateNotFasterThen()),
        config.GetMaxWaitAttempts(),
        TDuration::MilliSeconds(config.GetSleepBetweenWaitAttempts())
    );
}

NPq2Saas::TConfig LoadConfig(int argc, char** argv) {
    TString configPath;
    NLastGetopt::TOpts opts;
    opts.AddLongOption('c', "config", "Path to json config file")
        .DefaultValue("metrika_test")
        .StoreResult(&configPath);
    opts.AddLongOption("use-presets", "If set, config lookup will be done by archived ones")
        .NoArgument();
    opts.AddLongOption("print-config", "If set, prints config and exits")
        .NoArgument();
    opts.AddLongOption("mon-port", "If set, overrides monitoring port from config");
    opts.AddLongOption("loc", "If set, determines dc from location and overrides config");
    opts.AddLongOption("dc", "If set, overrides location option and config");
    opts.AddLongOption("skip-lag", "If set, overrides SkipInitialLag param from pq config")
        .NoArgument();
    opts.AddHelpOption();
    NLastGetopt::TOptsParseResult optsParser(&opts, argc, argv);

    NPq2Saas::InitConfig(configPath, optsParser.Has("use-presets"));
    auto config = NPq2Saas::GetConfig();
    if (optsParser.Has("mon-port")) {
        config.MutableMonitoring()->SetPort(optsParser.Get<ui32>("mon-port"));
    }
    if (optsParser.Has("loc")) {
        config.MutableCommon()->SetDC(ConvertLocationToDC(optsParser.Get<TString>("loc")));
    }
    if (optsParser.Has("dc")) {
        config.MutableCommon()->SetDC(optsParser.Get<TString>("dc"));
    }
    if (config.GetCommon().GetDC().empty()) {
        config.MutableCommon()->SetDC(GetCurrentDC());
    }
    if (optsParser.Has("skip-lag")) {
        config.MutablePQ()->SetSkipInitialLag(true);
    }
    if (optsParser.Has("print-config")) {
        Cout << config.AsJSON() << Endl;
        exit(1);
    }
    return config;
}

TVector<TString> GetLogbrokerServersUrls(const NPq2Saas::TConfig& config) {
    if (!config.GetPQ().GetServers().empty()) {
        return TVector<TString>(config.GetPQ().GetServers().begin(),
                                config.GetPQ().GetServers().end());
    }

    if (config.GetPQ().GetPrependDC()) {
        return {config.GetCommon().GetDC() + "." + config.GetPQ().GetServer()};
    }
    return {config.GetPQ().GetServer()};
}

NPq2Saas::TCallbackSettings CreateCallbackSettings(const NPq2Saas::TConfig& config) {
    const auto& parsing = config.GetQueues().GetParsing();
    const auto& sending = config.GetQueues().GetSending();
    NPq2Saas::TCallbackSettings result;
    result.DataCenter = config.GetCommon().GetDC();
    result.DataCenterAffinity.emplace("vla", "iva");
    result.DataCenterAffinity.emplace("sas", "myt");
    result.ParsingWorkersCount = parsing.GetWorkersCount();
    result.MaxParsingQueueSize = parsing.GetMaxInFlight();
    result.WorkersCount = sending.GetWorkersCount();
    result.MaxInFlight = sending.GetMaxInFlight();
    result.WaitSenderTimeout = TDuration::MilliSeconds(sending.GetWaitQueueAvailableTimeout());
    for (const auto& destination : config.GetDestinations()) {
        auto resource = ParseDestinationConfig(destination);
        result.Manager->AddResource(destination.GetName(), std::move(resource));
    }
    for (const auto& cache : config.GetCaches()) {
        auto resource = ParseCacheConfig(cache);
        result.Manager->AddResource(cache.GetName(), std::move(resource));
    }
    for (const auto& delivery : config.GetDeliveries()) {
        if (delivery.GetEnabled()) {
            auto insertResult = result.Deliveries.emplace(
                delivery.GetDeliveryName(), NPq2Saas::THandlerSettings(delivery.GetHandlerConfig())
            );
            Y_VERIFY(insertResult.second, "Each delivery should have unique name");
        }
    }
    for (const auto& kvPair : result.Deliveries) {
        if (config.GetLog().GetPrintSettings()) {
            Cerr << "= Delivery " << kvPair.first << Endl;
            kvPair.second.PrintToStream(Cerr);
            Cerr << Endl;
        }
        kvPair.second.Verify(result.Manager);
    }
    Y_VERIFY(result.Deliveries.size() > 0, "No active destinations set");
    result.SkipInitialLag = config.GetPQ().GetSkipInitialLag();
    result.HardCommitPeriod = TDuration::Seconds(config.GetPQ().GetHardCommitPeriod());
    result.MaxReservedBufferKbSize = config.GetPQ().GetMaxReservedBufferSize() * 1024;

    switch (config.GetPQ().GetLogFormat()) {
    case NPq2SaasProto::TPQConfig::TSKV:
        result.Format = NPq2Saas::TCallbackSettings::EFormat::Tskv;
        break;
    case NPq2SaasProto::TPQConfig::JSON:
        result.Format = NPq2Saas::TCallbackSettings::EFormat::Json;
        break;
    default:
        Y_FAIL("unimplemented");
        break;
    }

    return result;
}

NPersQueue2::TSettings CreatePQSettings(const NPq2Saas::TConfig& config) {
    const auto& pqConfig = config.GetPQ();
    NPersQueue2::TSettings result;
    result.Format = NPersQueue2::TSettings::FORMAT_RAW;
    result.Partition = -1; // read from random partition
    auto servers = GetLogbrokerServersUrls(config);
    Y_VERIFY(servers.size() == 1, "Only one server allowed");
    result.ServerHostname = servers[0];
    result.Topic = pqConfig.GetTopic();
    result.LogType = pqConfig.GetLogType();
    result.FileName = pqConfig.GetClientId();
    result.UseIpv6 = pqConfig.GetEnableIPv6();
    result.UseMirroredPartitions = pqConfig.GetUseMirroredPartitions();
    result.Transferable = pqConfig.GetTransferable();
    result.ReadTimeout = pqConfig.GetChunkReadTimeout();
    result.StreamRead = pqConfig.GetStreamRead();
    return result;
}

void LoadTraces(const NPq2Saas::TConfig& config, NLWTrace::TManager& manager) {
    Cerr << "[TRACES]" << Endl;
    for (const auto& trace : config.GetTraces()) {
        Cerr << "* Loading " << trace.GetName() << ":" << Endl;
        Y_VERIFY(trace.HasFileName() || trace.HasQuery(), "Trace should have either FileName or Query");
        NLWTrace::TQuery query;
        if (trace.HasQuery()) {
            query.CopyFrom(trace.GetQuery());
        } else {
            auto script = TUnbufferedFileInput(trace.GetFileName()).ReadAll();
            bool ok = google::protobuf::TextFormat::ParseFromString(script, &query);
            Y_VERIFY(ok, "Failed to parse lwtrace protobuf from file %s", trace.GetFileName().data());
        }
        Cerr << ' ' << query << Endl;
        manager.New(trace.GetName(), query);
    }
}

} // anonymous namespace

int MainDeprecated(const NPq2Saas::TConfig& config);
int MainNew(const NPq2Saas::TConfig& config);

int main(int argc, char** argv) {
    signal(SIGPIPE, SIG_IGN);

    const auto& config = LoadConfig(argc, argv);
    // Logging
    const auto& log = config.GetLog();
    NPq2Saas::TLogger::Setup(log.HasLogPath() ? MakeMaybe(log.GetLogPath()) : Nothing());
    if (const auto& globalLogPath = log.GetGlobalLogPath()) {
        DoInitGlobalLog(globalLogPath, log.GetGlobalLogLevel(), false, false);
    } else {
        NRTLine::InitializeLogging();
    }
    auto reopenLog = [](int) { NPq2Saas::TLogger::Reopen(); };
    SetAsyncSignalFunction(SIGUSR1, reopenLog);

    // Traces
    NPq2Saas::RegisterAllProbes(NLwTraceMonPage::ProbeRegistry());
    LoadTraces(config, NLwTraceMonPage::TraceManager());

    if (config.GetPQ().GetNewApi()) {
        return MainNew(config);
    } else {
        return MainDeprecated(config);
    }
}

int MainDeprecated(const NPq2Saas::TConfig& config) {
    // Monitoring
    TFnProxy fnProxy;
    auto& monitoring = NPq2SaasMonitoring::InitMonitoring(config.GetMonitoring().GetPort(),
        [&fnProxy] { fnProxy.Fn(); });

    // LogBroker Consumer
    NPq2Saas::TConsumerCallbackManager cbManager(CreateCallbackSettings(config));
    NPersQueue2::TCerrLogger logger(config.GetLog().GetPQLogLevel());
    const auto& pqConfig = config.GetPQ();
    NPersQueue2::TPQLib pq(nullptr, pqConfig.GetMaxPartitions());
    NPersQueue2::TConsumer consumer(pq,
        CreatePQSettings(config),
        pqConfig.GetChunkSize(),
        Max<ui32>()/*ByteSize*/,
        pqConfig.GetMaxPartitions(),
        cbManager.CreateCallback().Release(),
        pqConfig.GetTimeSleep(),
        pqConfig.GetTimeLag(),
        &logger);

    fnProxy.Fn = [&cbManager, &monitoring] {
        *monitoring.GetCounter("EventsSendingQueueSize") = cbManager.GetSendingQueueSize();
        *monitoring.GetCounter("EventsParsingQueueSize") = cbManager.GetParsingQueueSize();
    };

    // Starting
    monitoring.Start();
    Log("Starting...");
    consumer.Start();

    TManualEvent shutdownEvent;
    auto shutdownCb = [&shutdownEvent](int) { shutdownEvent.Signal(); };
    SetAsyncSignalFunction(SIGINT, shutdownCb);
    SetAsyncSignalFunction(SIGTERM, shutdownCb);

    Log("Everything is up and running...");
    shutdownEvent.WaitI();

    Log("Stopping...");
    cbManager.RequestStop();
    consumer.Stop();
    Log("Consumer stopped!");
    cbManager.Wait();
    Log("Workers queue in callback stopped!");
    monitoring.Shutdown();
    Log("Monitoring stopped!");
    Log("Bye-bye!");
    return EXIT_SUCCESS;
}

class TConsumerState {
private:
    const TString Server;
    const ui64 ReadTimestampMs;
    TIntrusivePtr<NPersQueue::TCerrLogger> Logger;
    NPersQueue::TPQLib& PQLib;
    const NPq2SaasProto::TPQConfig& PQConfig;

    THolder<NPersQueue::IConsumer> Consumer;
    TFuture<NPersQueue::TConsumerMessage> Future;
    TInstant FutureStart;

public:
    TConsumerState(const TString& server, NPersQueue::TPQLib& pq,
                   const NPq2SaasProto::TPQConfig& pqConfig,
                   TIntrusivePtr<NPersQueue::TCerrLogger> logger,
                   const TInstant startReadingTimestamp)
        : Server(server)
        , ReadTimestampMs(startReadingTimestamp.MilliSeconds())
        , Logger(logger)
        , PQLib(pq)
        , PQConfig(pqConfig)
    {
        RestartConsumer();
    }

    const TString& GetServer() const {
        return Server;
    }

    TInstant GetFutureStart() const {
        return FutureStart;
    }

    NPersQueue::IConsumer& GetConsumer() {
        Y_ENSURE(Consumer);
        return *Consumer;
    }

    const TFuture<NPersQueue::TConsumerMessage>& GetFuture() const {
        return Future;
    }

    void RestartConsumer() {
        auto consumer = StartConsumer();
        Y_ENSURE(consumer, "Failed to start consumer " << Server);
        Consumer = std::move(consumer);
        UpdateFuture();
    }

    void UpdateFuture() {
        Y_ENSURE(Consumer);
        Future = Consumer->GetNextMessage();
        FutureStart = TInstant::Now();
    }

private:
    THolder<NPersQueue::IConsumer> StartConsumer() {
        NPersQueue::TConsumerSettings consumerSettings;
        consumerSettings.Server.Address = Server;
        consumerSettings.Topics = { PQConfig.GetTopic() + '/' + PQConfig.GetLogType() };
        consumerSettings.ClientId = PQConfig.GetClientId();
        consumerSettings.ReadMirroredPartitions = PQConfig.GetUseMirroredPartitions();
        consumerSettings.ReadTimestampMs = ReadTimestampMs;
        consumerSettings.MaxSize = PQConfig.GetChunkSizeKb() * 1024;
        consumerSettings.MaxCount = PQConfig.GetChunkSize();
        consumerSettings.CredentialsProvider = NPersQueue::CreateTVMCredentialsProvider(
            GetEnv(PQConfig.GetTvmSecret(), PQConfig.GetTvmSecret()),
            PQConfig.GetTvmSelfClientId(),
            PQConfig.GetTvmDestinationClientId(),
            Logger
        );

        INFO_LOG << "Starting consumer for " << Server << Endl;
        auto consumer = PQLib.CreateConsumer(consumerSettings, Logger);
        if (!consumer) {
            ERROR_LOG << "cannot create consumer" << Endl;
            return nullptr;
        }

        auto waitTimeout = TDuration::MilliSeconds(PQConfig.GetFutureWaitTimeout());
        auto starter = consumer->Start();
        if (!starter.Wait(waitTimeout)) {
            ERROR_LOG << "starter wait timeout" << Endl;
            return nullptr;
        }
        if (starter.GetValue().Response.HasError()) {
            ERROR_LOG << "error while starting consumer: "
                      << starter.GetValue().Response.GetError().GetDescription() << Endl;
            return nullptr;
        }
        return consumer;
    }
};

TVector<const TFuture<NPersQueue::TConsumerMessage>> GetFutures(
        const TVector<TConsumerState>& consumers) {
    TVector<const TFuture<NPersQueue::TConsumerMessage>> futures(Reserve(consumers.size()));
    for (auto&& c : consumers) {
        futures.push_back(c.GetFuture());
    }
    return futures;
}

int MainNew(const NPq2Saas::TConfig& config) {
    auto active = TAtomic(1);
    auto shutter = [&active](int) {
        AtomicSet(active, 0);
    };
    SetAsyncSignalFunction(SIGINT, shutter);
    SetAsyncSignalFunction(SIGTERM, shutter);

    // Monitoring
    TFnProxy functionProxy;
    auto& monitoring = NPq2SaasMonitoring::InitMonitoring(config.GetMonitoring().GetPort(), [&functionProxy] {
        functionProxy.Fn();
    });

    NPq2Saas::TConsumerCallbackManager consumerCallbackManager(CreateCallbackSettings(config));
    functionProxy.Fn = [&consumerCallbackManager, &monitoring] {
        *monitoring.GetCounter("EventsSendingQueueSize") = consumerCallbackManager.GetSendingQueueSize();
        *monitoring.GetCounter("EventsParsingQueueSize") = consumerCallbackManager.GetParsingQueueSize();
    };
    TMonitoringGuard monitoringGuard(monitoring);

    TConsumerCallbackManagerGuard consumerCallbackManagerGuard(consumerCallbackManager);
    auto baseCallback = consumerCallbackManager.CreateCallback();
    if (!baseCallback) {
        ERROR_LOG << "cannot create base callback" << Endl;
        return EXIT_FAILURE;
    }

    const auto& pqConfig = config.GetPQ();
    auto logger = MakeIntrusive<NPersQueue::TCerrLogger>(static_cast<int>(config.GetLog().GetPQLogLevel()));
    auto waitTimeout = TDuration::MilliSeconds(pqConfig.GetFutureWaitTimeout());

    auto serversUrls = GetLogbrokerServersUrls(config);
    Y_VERIFY(!serversUrls.empty(), "Missing lb server");

    NPersQueue::TPQLibSettings settings;
    settings.ThreadsCount = pqConfig.GetThreadsCount();
    NPersQueue::TPQLib pq(settings);

    TInstant startReadingTimestamp;
    if (pqConfig.GetSkipInitialLag()) {
        startReadingTimestamp = TInstant::Now();
    }

    TVector<TConsumerState> consumers(Reserve(serversUrls.size()));
    for (auto&& s : serversUrls) {
        consumers.push_back(TConsumerState(s, pq, pqConfig, logger, startReadingTimestamp));
    }

    while (active) {
        TInstant now = TInstant::Now();
        auto trackId = baseCallback->NextIteration();
        GLOBAL_LWPROBE(PQ2SAAS_CONSUMER_PROVIDER, Iteration_Begin, trackId);
        NThreading::WaitAny(GetFutures(consumers)).Wait(waitTimeout);
        GLOBAL_LWPROBE(PQ2SAAS_CONSUMER_PROVIDER, LBMessageFetch, trackId);
        for (auto&& c : consumers) {
            if (c.GetFuture().HasException()) {
                ERROR_LOG << c.GetServer() << " consumer error: "
                          << NThreading::GetExceptionMessage(c.GetFuture()) << Endl;
                c.RestartConsumer();
                continue;
            }

            if (!c.GetFuture().HasValue()) {
                if (now - c.GetFutureStart() > waitTimeout) {
                    ERROR_LOG << c.GetServer() << " consumer wait timeout" << Endl;
                    c.RestartConsumer();
                }
                continue;
            }

            const auto& value = c.GetFuture().GetValue();
            switch (value.Type) {
            case NPersQueue::EMT_ERROR:
            {
                ERROR_LOG << c.GetServer() << " consumer error while getting next message: "
                          << value.Response.GetError().GetDescription() << Endl;
                c.RestartConsumer();
                continue;
            }
            case NPersQueue::EMT_DATA:
            {
                for (const auto& t : value.Response.data().message_batch()) {
                    auto dead = NThreading::MakeFuture();
                    for (const auto& m : t.message()) {
                        const auto offset = static_cast<NPersQueue2::TConsumerRecId>(m.offset());
                        const TString& data = m.data();
                        TBuffer buf(data.data(), data.size());
                        NPersQueue2::TProducerInfo info;
                        bool softCommit = false;
                        baseCallback->WaitForSender();
                        baseCallback->OnData(&buf, info, 1, offset, &softCommit);
                    }
                }

                auto cookie = value.Response.GetData().GetCookie();
                c.GetConsumer().Commit({ cookie });
                break;
            }
            case NPersQueue::EMT_COMMIT:
                INFO_LOG << "committed cookie: " << JoinSeq(",", value.Response.GetCommit().GetCookie()) << Endl;
                break;
            default:
                ERROR_LOG << "unexpected message type: " << static_cast<int>(value.Type) << Endl;
                break;
            }
            c.UpdateFuture();
        }
        GLOBAL_LWPROBE(PQ2SAAS_CONSUMER_PROVIDER, Iteration_End, trackId);
    }
    return EXIT_SUCCESS;
}
