#include "auth.h"
#include "config.h"
#include "tvm.h"


#include <drive/backend/auth/fake/fake.h>

#include <drive/library/cpp/tvm/logger.h>

namespace NDrive::NPumpkin {

    THolder<IAuthModule> IAuthManager::ConstructAuthModule() const {
        return ConstructAuthModuleImpl();
    }

    TBlackboxAuthManager::TBlackboxAuthManager(const NDrive::NPumpkin::NProto::TBlackboxAuthManagerConfig& config)
        : Config(config)
        , Blackbox2AuthConfig(GetBlackbox2AuthConfig(Config.GetBlackboxAuthConfig()))
    {
        TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient;
        if (config.HasTvmConfig()) {
            tvmClient = BuildTvmClient(config.GetTvmConfig());
        }
        BlackboxClient = BuildBlackboxClient(config.GetBlackboxAuthConfig(), tvmClient);
    }

    THolder<IAuthModule> TBlackboxAuthManager::ConstructAuthModuleImpl() const {
        return MakeHolder<TBlackbox2AuthModule>(Blackbox2AuthConfig, BlackboxClient);
    }

    TAtomicSharedPtr<NDrive::TBlackboxClient> TBlackboxAuthManager::BuildBlackboxClient(const NDrive::NPumpkin::NProto::TBlackboxAuthConfig& config, TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient) const {
        auto client = MakeAtomicShared<NDrive::TBlackboxClient>(config.GetUrl(), tvmClient);
        client->SetExternalOptions(NBlackbox2::TOptions{});
        if (const auto& cookieHost = config.GetCookieHost()) {
            client->SetCookieHost(cookieHost);
        }
        if (const auto& destinationClientId = config.GetDestinationClientId()) {
            client->SetDestinationClientId(destinationClientId);
        }
        {
            TVector<TString> requiredScopes;
            for (const auto& scope : config.GetRequiredScopes()) {
                requiredScopes.push_back(scope);
            }
            if (!requiredScopes.empty()) {
                client->SetScopes(requiredScopes);
            }
        }
        if (config.GetEnableUserTickets()) {
            client->EnableUserTickets();
        }
        return client;
    }


    THolder<IAuthModule> TFakeAuthManager::ConstructAuthModuleImpl() const {
        return MakeHolder<TFakeAuthModule>(nullptr);
    }

    THolder<IAuthManager> BuildAuthManager(const NDrive::NPumpkin::NProto::TAuthManagerConfig& config) {
        if (config.GetType() == "fake") {
            return MakeHolder<TFakeAuthManager>();
        } else if (config.GetType() == "blackbox") {
            Y_ENSURE(config.HasBlackboxAuthManagerConfig());
            return MakeHolder<TBlackboxAuthManager>(config.GetBlackboxAuthManagerConfig());
        }
        return nullptr;
    }
}
