#include "cars_storage.h"

#include <rtline/library/json/cast.h>

template<>
NJson::TJsonValue NJson::ToJson(const NDrive::NPumpkin::TSimpleCarData& carData) {
    return TMapBuilder
            ("id", carData.GetId())
            ("imei", carData.GetIMEI())
            ("number", carData.GetNumber());
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::NPumpkin::TSimpleCarData& result) {
    return NJson::ParseField(value["id"], result.MutableId())
        && NJson::ParseField(value["imei"], result.MutableIMEI())
        && NJson::ParseField(value["number"], result.MutableNumber());
}

namespace NDrive::NPumpkin {
    TSimpleCarsDataStorage::TSimpleCarsDataStorage(TVector<TSimpleCarData>&& data)
        : CarsData(std::move(data))
    {
        for (const auto& car : CarsData) {
            CarsById.push_back(std::cref(car));
            CarsByIMEI.push_back(std::cref(car));
            CarsByNumber.push_back(std::cref(car));
        }
        std::sort(CarsById.begin(), CarsById.end(), [](const auto& lhs, const auto& rhs){
            return lhs.get().GetId() < rhs.get().GetId();
        });
        std::sort(CarsByIMEI.begin(), CarsByIMEI.end(), [](const auto& lhs, const auto& rhs){
            return lhs.get().GetIMEI() < rhs.get().GetIMEI();
        });
        std::sort(CarsByNumber.begin(), CarsByNumber.end(), [](const auto& lhs, const auto& rhs){
            return lhs.get().GetNumber() < rhs.get().GetNumber();
        });
    }

    TSet<TSimpleCarData> TSimpleCarsDataStorage::FindCarsByPrefix(TStringBuf prefix, ui32 limit) const {
        TSet<TSimpleCarData> res;
        {
            auto start = std::lower_bound(CarsById.begin(), CarsById.end(), prefix, [](const auto& dataRef, const TStringBuf p) {
                return dataRef.get().GetId() < p;
            });
            for (auto it = start; it != CarsById.end() && it->get().GetId().StartsWith(prefix) && res.size() < limit; ++it) {
                res.insert(*it);
            }
        }
        {
            auto start = std::lower_bound(CarsByIMEI.begin(), CarsByIMEI.end(), prefix, [](const auto& dataRef, const TStringBuf p) {
                return dataRef.get().GetIMEI() < p;
            });
            for (auto it = start; it != CarsByIMEI.end() && it->get().GetIMEI().StartsWith(prefix) && res.size() < limit; ++it) {
                res.insert(*it);
            }
        }
        {
            auto start = std::lower_bound(CarsByNumber.begin(), CarsByNumber.end(), prefix, [](const auto& dataRef, const TStringBuf p) {
                return dataRef.get().GetNumber() < p;
            });
            for (auto it = start; it != CarsByNumber.end() && it->get().GetNumber().StartsWith(prefix) && res.size() < limit; ++it) {
                res.insert(*it);
            }
        }
        return res;
    }

    TMaybe<TSimpleCarData> TSimpleCarsDataStorage::GetCarByNumber(TStringBuf number) const {
        auto it = std::lower_bound(CarsByNumber.begin(), CarsByNumber.end(), number, [](const auto& dataRef, const TStringBuf number) {
            return dataRef.get().GetNumber() < number;
        });
        return (it != CarsByNumber.end() && it->get().GetNumber() == number) ? MakeMaybe<TSimpleCarData>(it->get()) : Nothing();
    }

    TMaybe<TSimpleCarData> TSimpleCarsDataStorage::GetCarByIMEI(TStringBuf imei) const {
        auto it = std::lower_bound(CarsByIMEI.begin(), CarsByIMEI.end(), imei, [](const auto& dataRef, const TStringBuf imei) {
            return dataRef.get().GetIMEI() < imei;
        });
        return (it != CarsByIMEI.end() && it->get().GetIMEI() == imei) ? MakeMaybe<TSimpleCarData>(it->get()) : Nothing();
    }

    const TVector<TSimpleCarData>& TSimpleCarsDataStorage::GetCarsData() const {
        return CarsData;
    }

    TVector<TString> TSimpleCarsDataStorage::GetCarsIMEIs() const {
        TVector<TString> imeis;
        imeis.reserve(CarsData.size());
        for (const auto& car: CarsData) {
            imeis.push_back(car.GetIMEI());
        }
        return imeis;
    }
}
