#include "config.h"

#include "json_text_patcher.h"
#include "tvm.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/protobuf/json/json2proto.h>

#include <util/folder/path.h>
#include <util/stream/file.h>

namespace NDrive::NPumpkin {
    TConfig::TConfig(const TFsPath& path, TLuaJsonTextPatcher* patcher) {
        NJson::TJsonValue json;
        TFileInput file(path);
        if (!patcher) {
            NJson::ReadJsonTree(&file, true, &json, true);
        } else {
            TString patchedConfig;
            TStringOutput output{patchedConfig};
            patcher->Patch(file, output);
            NJson::ReadJsonFastTree(patchedConfig, &json, /* throwOnError= */ true);
        }
        Cerr << json.GetStringRobust() << Endl;
        NProtobufJson::Json2Proto(json, *this);
        // load secrets
        if (auto file = GetAuthManagerConfig().GetBlackboxAuthManagerConfig().GetTvmConfig().GetSecretFile()) {
            MutableAuthManagerConfig()->MutableBlackboxAuthManagerConfig()->MutableTvmConfig()->SetSecret(TIFStream(file).ReadAll());
        }
        if (auto file = GetTelematicsClientConfig().GetTvmConfig().GetSecretFile()) {
            MutableTelematicsClientConfig()->MutableTvmConfig()->SetSecret(TIFStream(file).ReadAll());
        }
    }

    THttpServerOptions GetHttpOptions(const NDrive::NPumpkin::NProto::THttpConfig& config) {
        THttpServerOptions result;
        result.Port = config.GetPort();
        result.nThreads = config.GetThreads();
        return result;
    }

    TBlackbox2AuthConfig GetBlackbox2AuthConfig(const NDrive::NPumpkin::NProto::TBlackboxAuthConfig& config) {
        TBlackbox2AuthConfig result("");
        result.SetUrl(config.GetUrl());
        result.SetCookieHost(config.GetCookieHost());
        result.SetEnvironment(config.GetEnvironment());
        {
            TSet<TString> acceptedScopes;
            for (const auto& scope : config.GetAcceptedScopes()) {
                acceptedScopes.insert(scope);
            }
            result.SetAcceptedScopes(acceptedScopes);
        }
        {
            TVector<TString> requiredScopes;
            for (const auto& scope : config.GetRequiredScopes()) {
                requiredScopes.push_back(scope);
            }
            result.SetRequiredScopes(requiredScopes);
        }
        result.SetDestinationClientId(config.GetDestinationClientId());
        result.SetSelfClientId(config.GetSelfClientId());
        result.SetAuthMethod(FromString(config.GetAuthMethod()));
        result.SetIgnoreDeviceId(config.GetIgnoreDeviceId());
        result.SetEnableUserTickets(config.GetEnableUserTickets());
        return result;
    }

    TVector<TString> GetTelematicsClientShards(const NDrive::NPumpkin::NProto::TTelematicsClientOptions& config) {
        TVector<TString> result;
        for (const auto& shard : config.GetShards()) {
            result.push_back(shard);
        }
        return result;
    }

    NDrive::TTelematicsClient::TOptions GetTelematicsClientOptions(const NDrive::NPumpkin::NProto::TTelematicsClientOptions& config) {
        NDrive::TTelematicsClient::TOptions options;
        options.DestinationClientId = config.GetDestinationClientId();
        return options;
    }

    std::pair<TRTLineAPIConfig, TAtomicSharedPtr<NTvmAuth::TTvmClient>> GetRTLineAPIConfig(const NDrive::NPumpkin::NProto::TRTLineAPIConfig& config) {
        TRTLineAPIConfig result;
        TAtomicSharedPtr<NTvmAuth::TTvmClient> client;
        result.SetHost(config.GetHost());
        result.SetPort(config.GetPort());
        result.SetServiceName(config.GetServiceName());
        if (config.BalancerTimeoutTableMsSize()) {
            result.MutableBalancerOptions().BalancerTimeoutTable.clear();
            for (const ui32& timeout : config.GetBalancerTimeoutTableMs()) {
                result.MutableBalancerOptions().BalancerTimeoutTable.push_back(TDuration::MilliSeconds(timeout));
            }
        }
        result.MutableSearchRequestConfig() = NSimpleMeta::TConfig::ForRequester();
        if (config.HasSearchRequestConfig()) {
            result.MutableSearchRequestConfig().SetGlobalTimeout(TDuration::MilliSeconds(config.GetSearchRequestConfig().GetGlobalTimeout()))
                                           .SetConnectTimeout(TDuration::MilliSeconds(config.GetSearchRequestConfig().GetConnectTimeout()))
                                           .SetTasksCheckInterval(TDuration::MilliSeconds(config.GetSearchRequestConfig().GetTasksCheckIntervalMs()))
                                           .SetMaxAttempts(config.GetSearchRequestConfig().GetMaxAttempts());
        }

        if (config.HasTvmConfig()) {
            client = BuildTvmClient(config.GetTvmConfig());
        }
        if (config.HasDestinationTvmId()) {
            result.SetDestinationTvmId(config.GetDestinationTvmId());
        }
        return { result, client };
    }
}
