#include "logging.h"

#include "config.h"

#include <drive/backend/logging/events.h>
#include <drive/backend/logging/logging.h>

#include <drive/library/cpp/logger/formatter.h>

#include <library/cpp/logger/global/global.h>
#include <logbroker/unified_agent/client/cpp/logger/backend.h>

#include <rtline/library/unistat/log/backend.h>

#include <util/folder/path.h>


namespace {
    void InitLogs(const NDrive::NPumpkin::TConfig& config) {
        DoInitGlobalLog(config.GetLoggingConfig().GetLoggerType(),
                        config.GetLoggingConfig().GetLogLevel(),
                        config.GetLoggingConfig().GetLogRotation(),
                        false);
        const TString& stdout = config.GetLoggingConfig().GetStdOut();
        const TString& stderr = config.GetLoggingConfig().GetStdErr();
        if (stdout && !TFsPath(stdout).Parent().Exists()) {
            ythrow yexception() << "directory with stdout redirect file is incorrect: " << TFsPath(stdout).Parent().GetPath();
        }
        if (stderr && !TFsPath(stderr).Parent().Exists()) {
            ythrow yexception() << "directory with stderr redirect file is incorrect: " << TFsPath(stderr).Parent().GetPath();
        }
        NDrive::NPumpkin::ReplaceDescriptor(stdout, STDOUT_FILENO);
        NDrive::NPumpkin::ReplaceDescriptor(stderr, STDERR_FILENO);
    }
}

namespace NDrive::NPumpkin {
    void ReplaceDescriptor(const TString& name, FHANDLE fd) {
        if (!name || name == TStringBuf("console") || name == TStringBuf("null")) {
            return;
        }
        TFileHandle stream(fd);
        TFileHandle fileStream{name, OpenAlways | WrOnly | ForAppend | AWUser | ARUser | ARGroup | AROther};
        Y_ENSURE(fileStream.IsOpen(), "can't open " << name << ": " << strerror(errno));
        Y_ENSURE(stream.LinkTo(fileStream),
            TStringBuilder() << "can't link " << name << " to " << static_cast<int>(fd) << ": " << strerror(errno));
        stream.Release();
    }

    void InitLogging(const TConfig& config) {
        if (const auto& globalLog = config.GetLoggingConfig().GetLoggerType(); globalLog || !GlobalLogInitialized()) {
            DoInitGlobalLog(MakeHolder<TUnistatLogBackend>(
                globalLog,
                static_cast<ELogPriority>(config.GetLoggingConfig().GetLogLevel())
            ), MakeHolder<NDrive::TLoggerFormatter>());
        }
        if (const TString& eventLog = config.GetLoggingConfig().GetEventLog()) {
            NLoggingImpl::InitLogImpl<NDrive::TEventLog>(eventLog, LOG_MAX_PRIORITY, config.GetLoggingConfig().GetLogRotation(), /* StartAsDaemon */ false);
        }
        if (config.GetLoggingConfig().HasUnifiedAgentConfig()) {
            const auto& unifiedAgentConfig = config.GetLoggingConfig().GetUnifiedAgentConfig();
            auto parameters = NUnifiedAgent::TClientParameters(unifiedAgentConfig.GetEventLog());
            if (TLoggerOperator<NDrive::TEventLog>::Usage()) {
                parameters.SetLog(TLoggerOperator<NDrive::TEventLog>::Log());
            }
            if (unifiedAgentConfig.HasGrpcMaxMessageSize()) { // uint32 casts to size_t
                parameters.SetGrpcMaxMessageSize(unifiedAgentConfig.GetGrpcMaxMessageSize());
            }
            if (auto grpcReconnectDelay = TDuration::Parse(unifiedAgentConfig.GetGrpcReconnectDelay())) {
                parameters.SetGrpcReconnectDelay(grpcReconnectDelay);
            }
            if (auto grpcSendDelay = TDuration::Parse(unifiedAgentConfig.GetGrpcSendDelay())) {
                parameters.SetGrpcSendDelay(grpcSendDelay);
            }
            if (unifiedAgentConfig.HasMaxInflightBytes()) { // uint32 casts to size_t
                parameters.SetMaxInflightBytes(unifiedAgentConfig.GetMaxInflightBytes());
            }
            auto backend = NUnifiedAgent::MakeLogBackend(parameters);
            TLoggerOperator<NDrive::TEventLog>::Set(new NDrive::TEventLog(std::move(backend)));
        }
        NDrive::TUnistatSignals::RegisterFallback(); // TODO: FWIW?
        InitLogs(config);
    }
}
