#include "pumpkin.h"

#include "auth.h"
#include "cars_storage.h"
#include "config.h"
#include "http_client.h"
#include "logging.h"
#include "tvm.h"

#include <drive/library/cpp/searchserver/server.h>
#include <drive/telematics/api/sensor/client.h>
#include <drive/telematics/api/sensor/local.h>


namespace NDrive {
    NDrive::NPumpkin::TConfig TPumpkin::Config;

    void TPumpkin::SetConfig(const NDrive::NPumpkin::TConfig& config) {
        Config = config;
    }

    TPumpkin::TPumpkin()
        : TSearchServerBase(NDrive::NPumpkin::GetHttpOptions(Config.GetHttpConfig()))
        , AuthManager(NDrive::NPumpkin::BuildAuthManager(Config.GetAuthManagerConfig()))
    {
        HandleSignals();
        InitSimpleCarsDataStorage(Config.GetSimpleCarsDataStorageConfig());
        InitRTLineAPIs(Config.GetRTLineAPIConfigs());
        InitSensorAPI();
        InitSensorsStorage(SensorAPI);
        InitUsersPermissionsStorage(Config.GetUsersPermissionsStorageConfig());
        TAtomicSharedPtr<NDrive::IServiceDiscovery> sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(
            NDrive::NPumpkin::GetTelematicsClientShards(Config.GetTelematicsClientConfig().GetTelematicsClientOptions())
        );
        TAtomicSharedPtr<NTvmAuth::TTvmClient> tvmClient = nullptr;
        if (Config.GetTelematicsClientConfig().HasTvmConfig()) {
            tvmClient = NDrive::NPumpkin::BuildTvmClient(Config.GetTelematicsClientConfig().GetTvmConfig());
        }
        TelematicsClient = MakeHolder<TTelematicsClient>(
            sd,
            NDrive::NPumpkin::GetTelematicsClientOptions(Config.GetTelematicsClientConfig().GetTelematicsClientOptions()),
            tvmClient
        );
    }

    void TPumpkin::Run() {
        try {
            INFO_LOG << "Starting server" << Endl;
            TSearchServerBase::Run();
        } catch (const std::exception& e) {
            FATAL_LOG << "Starting failed: " << FormatExc(e) << Endl;
        }
    }

    void TPumpkin::Wait() {
        TSearchServerBase::Wait();
    }

    void TPumpkin::Stop() {
        // ??? SigHandlerThread->Stop();
        // ??? SigHandlerThread.Destroy();
        TGuard<TMutex> guard(Mutex);
        if (!Stopped) {
            Stopped = true;
            TelematicsClient.Reset();
            TSearchServerBase::Stop();
        }
    }


    void TPumpkin::ReopenLog(int) {
        DEBUG_LOG << "start loggers reopen...";
        NDrive::NPumpkin::ReplaceDescriptor(Config.GetLoggingConfig().GetStdOut(), STDOUT_FILENO);
        NDrive::NPumpkin::ReplaceDescriptor(Config.GetLoggingConfig().GetStdErr(), STDERR_FILENO);
        TLogBackend::ReopenAllBackends(); // ???
        DEBUG_LOG << "loggers reopen finished";
    }


    void TPumpkin::StopServer(int signal) {
        TGuard<TMutex> guard(Mutex);
        if (!Stopped) {
            Stopped = true;
            INFO_LOG << "Signal caught, stopping server" << Endl;
            ui32 rigidStopLevel = ((signal == SIGINT) || (signal == SIGTERM)) ? Max<ui32>() : 0;
            if (rigidStopLevel) {
                TSearchServerBase::Stop();
            } else {
                TSearchServerBase::Shutdown();
            }
        } else {
            INFO_LOG << "Server stop already is in progress, ignoring signal" << Endl;
        }
    }

    void TPumpkin::StopServerStatic(int signal) {
        Singleton<TPumpkin>()->StopServer(signal);
    }

    void TPumpkin::AbortOnSignal(int sig) {
        FormatBackTrace(&SINGLETON_GENERIC_LOG_CHECKED(TGlobalLog, TRTYLogPreprocessor, TLOG_CRIT, "CRITICAL_INFO"));
        signal(sig, SIG_DFL);
        Y_FAIL("Aborting on signal %d", sig);
    }

    void TPumpkin::AbortOnException() {
        FormatBackTrace(&SINGLETON_GENERIC_LOG_CHECKED(TGlobalLog, TRTYLogPreprocessor, TLOG_CRIT, "CRITICAL_INFO"));
        Y_FAIL("Aborting on exception %s", CurrentExceptionMessage().c_str());
    }

    void TPumpkin::ReopenLogStatic(int signal) {
        Singleton<TPumpkin>()->ReopenLog(signal);
    }

    void TPumpkin::HandleSignals() {
        std::set_terminate(AbortOnException);
        TVector<NUtil::TSigHandler> sigHandlers;
        sigHandlers.push_back(NUtil::TSigHandler(SIGSEGV, AbortOnSignal));
        sigHandlers.push_back(NUtil::TSigHandler(SIGINT,  StopServerStatic));
        sigHandlers.push_back(NUtil::TSigHandler(SIGTERM, StopServerStatic));
        sigHandlers.push_back(NUtil::TSigHandler(SIGUSR1, StopServerStatic));
        sigHandlers.push_back(NUtil::TSigHandler(SIGHUP, ReopenLogStatic));
        SigHandlerThread = MakeHolder<NUtil::TSigHandlerThread>(sigHandlers);
        SigHandlerThread->DelegateSignals();
    }

    TClientRequest* TPumpkin::CreateClient() {
        return new NDrive::NPumpkin::THttpClient(*this);
    }

    const NDrive::NPumpkin::IAuthManager& TPumpkin::GetAuthManager() const {
        return *Yensured(AuthManager.Get());
    }

    const NDrive::TTelematicsClient& TPumpkin::GetTelematicsClient() const {
        return *Yensured(TelematicsClient.Get());
    }

    const ISensorApi& TPumpkin::GetSensorAPI() const {
        return *Yensured(SensorAPI.Get());
    }

    const NDrive::NPumpkin::TSensorsStorage& TPumpkin::GetSensorsStorage() const {
        return *Yensured(SensorsStorage.Get());
    }

    ui32 TPumpkin::GetPort() const {
        return Config.GetHttpConfig().GetPort();
    }

    TDuration TPumpkin::GetSensorsAPIQueryTimeout() const {
        return TDuration::MilliSeconds(500);
    }

    const NDrive::NPumpkin::TSimpleCarsDataStorage& TPumpkin::GetSimpleCarsDataStorage() const {
        return *Yensured(SimpleCarsDataStorage.Get());
    }

    const NDrive::NPumpkin::TUsersPermissionsStorage& TPumpkin::GetUsersPermissionsStorage() const {
        return *Yensured(UsersPermissionsStorage.Get());
    }

    void TPumpkin::InitSimpleCarsDataStorage(const NDrive::NPumpkin::NProto::TSimpleCarsDataStorageConfig& config) {
        TVector<NDrive::NPumpkin::TSimpleCarData> carsData;
        if (config.GetCarsDataFile()) {
            NJson::TJsonValue json;
            TFileInput file(config.GetCarsDataFile());
            NJson::ReadJsonTree(&file, true, &json, true);
            Y_ENSURE(NJson::TryFromJson(json, carsData));
        }
        SimpleCarsDataStorage = MakeHolder<NDrive::NPumpkin::TSimpleCarsDataStorage>(std::move(carsData));
    }

    void TPumpkin::InitRTLineAPIs(const ::google::protobuf::RepeatedPtrField<::NDrive::NPumpkin::NProto::TRTLineAPIConfig>& protoConfigs) {
        for (const auto& protoConfig : protoConfigs) {
            const auto [config, tvmClient] = NDrive::NPumpkin::GetRTLineAPIConfig(protoConfig);
            TMaybe<NDrive::TTvmAuth> tvmAuth = Nothing();
            if (tvmClient) {
                tvmAuth = MakeMaybe<NDrive::TTvmAuth>(tvmClient, config.GetDestinationTvmId());
            }
            auto api = TRTLineAPI(config, std::move(tvmAuth));
            RTLineAPIs.emplace(protoConfig.GetName(), std::move(api));
        }
    }

    void TPumpkin::InitSensorAPI() {
        if (Config.HasSensorAPIConfig()) {
            const auto& apiName = Config.GetSensorAPIConfig().GetRTLineAPIName();
            auto it = RTLineAPIs.find(apiName);
            Y_ENSURE(it != RTLineAPIs.end(), "no RTLineAPI with name " << apiName);
            const auto& search = it->second.GetSearchClient();
            if (Config.GetSensorAPIConfig().HasOptions()) {
                const auto& protoOptions = Config.GetSensorAPIConfig().GetOptions();
                TSensorApi::TOptions options;
                options.ReqClass = protoOptions.GetReqClass();
                options.Timeout = TDuration::MilliSeconds(protoOptions.GetTimeoutMs());
                options.HeavyTimeout = TDuration::MilliSeconds(protoOptions.GetHeavyTimeoutMs());
                options.MaxKeysPerQuery = protoOptions.GetMaxKeysPerQuery();
                SensorAPI = MakeHolder<TSensorApi>(apiName, search, options);
            } else {
                SensorAPI = MakeHolder<TSensorApi>(apiName, search);
            }
        } else {
            SensorAPI = MakeHolder<TLocalSensorApi>();
        }
    }

    void TPumpkin::InitSensorsStorage(TAtomicSharedPtr<ISensorApi> sensorAPI) {
        Y_ENSURE(sensorAPI);
        auto updateInterval = TDuration::Seconds(Config.GetSensorsStorageConfig().GetUpdateIntervalSeconds());
        SensorsStorage = MakeHolder<NDrive::NPumpkin::TSensorsStorage>(sensorAPI, updateInterval);
    }

    void TPumpkin::InitUsersPermissionsStorage(const NDrive::NPumpkin::NProto::TUsersPermissionsStorageConfig& config) {
        TVector<NDrive::NPumpkin::TUserPermissions> usersPermissions;
        if (config.GetUsersPermissionsFile()) {
            NJson::TJsonValue json;
            TFileInput file(config.GetUsersPermissionsFile());
            NJson::ReadJsonTree(&file, true, &json, true);
            Y_ENSURE(NJson::TryFromJson(json, usersPermissions));
        }
        UsersPermissionsStorage = MakeHolder<NDrive::NPumpkin::TUsersPermissionsStorage>(std::move(usersPermissions));
    }
}
