#include "replier.h"

#include "config.h"
#include "permissions.h"
#include "pumpkin.h"

#include <drive/backend/auth/blackbox2/auth.h>
#include <drive/backend/auth/fake/fake.h>
#include <drive/library/cpp/reqid/reqid.h>
#include <drive/library/cpp/searchserver/context/replier.h>
#include <drive/library/cpp/threading/future.h>
#include <drive/telematics/api/server/client.h>
#include <drive/telematics/protocol/actions.h>
#include <drive/telematics/protocol/vega.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/types/exception.h>

namespace {
    void SendReply(IReplyContext::TPtr context, NJson::TJsonValue&& reply, ui32 code) {
        TBufferOutput output;
        output << reply.GetStringRobust() << Endl;
        context->MakeSimpleReply(output.Buffer(), code);
        NDrive::TEventLog::Log(NDrive::TEventLog::Response, *context, code, std::move(reply));
    }

    TMaybe<TString> GetUserLogin(IAuthInfo::TPtr authInfo) {
        if (const auto* blackboxAuthInfo = dynamic_cast<const TBlackboxAuthInfo*>(authInfo.Get()); blackboxAuthInfo) {
            return blackboxAuthInfo->GetLogin();
        }
        if (const auto* fakeAuthInfo = dynamic_cast<const TFakeAuthInfo*>(authInfo.Get()); fakeAuthInfo) {
            return fakeAuthInfo->GetUserId();
        }
        return Nothing();
    }

    const TStringBuf ReqIdClass = "YDP";
}

namespace NDrive::NPumpkin {
    TReplier::TReplier(const NDrive::TPumpkin& pumpkin, IReplyContext::TPtr context)
        : IHttpReplier(context, nullptr)
        , Pumpkin(pumpkin)
        , Context(context)
    {
        NDrive::FormReqId(Context->MutableCgiParameters(), &Context->GetRequestData(), ReqIdClass);
        NDrive::TEventLog::TContextGuard contextGuard(Context.Get());
        NDrive::TEventLog::Log(NDrive::TEventLog::Access, *Context);
    }

    IAuthInfo::TPtr TReplier::GetAuthInfo() const {
        const auto& authManager = Pumpkin.GetAuthManager();
        THolder<IAuthModule> module = authManager.ConstructAuthModule();
        if (!module) {
            return nullptr;
        }
        return module->RestoreAuthInfo(Context);
    }

    IThreadPool* TReplier::DoSelectHandler() {
        return nullptr;
    }

    void TReplier::OnRequestExpired() {
    }

    TDuration TReplier::GetDefaultTimeout() const {
        return TDuration::Seconds(100);
    }

    void TReplier::DoSearchAndReply() {
        NDrive::TEventLog::TContextGuard contextGuard(Context.Get());
        NDrive::TEventLog::TUserIdGuard pendingUserIdGuard;
        auto authInfo = GetAuthInfo();
        if (!authInfo || !authInfo->IsAvailable()) {
            ui32 code = authInfo ? authInfo->GetCode() : 500;
            TString message = authInfo ? authInfo->GetMessage() : "";
            SendReply(NJson::TMapBuilder("error", "unauthorized")
                                        ("message", message)
                                        ("code", code), code);
            return;
        }
        auto login = GetUserLogin(authInfo);
        if (!login) {
            SendReply(NJson::TMapBuilder("error", "can't fetch login"), HTTP_FORBIDDEN);
            return;
        }
        NDrive::TEventLog::TUserIdGuard::Set(*login);
        NDrive::TEventLog::Log(NDrive::TEventLog::Authorization, *Context, NJson::TMapBuilder("auth", authInfo->GetInfo()));
        auto permissions = Pumpkin.GetUsersPermissionsStorage().GetUserPermissions(*login);
        if (!permissions) {
            SendReply(NJson::TMapBuilder("error", "forbidden"), HTTP_FORBIDDEN);
            return;
        }

        const auto post = GetPost();
        const TStringBuf handle = Yensured(Context)->GetUri();
        const auto& cgi = Yensured(Context)->GetCgiParameters();
        if (handle == "/api/staff/car/control") {
            CarControl(std::move(post), *permissions);
        } else if (handle == "/api/staff/car/search") {
            CarSearch(std::move(post));
        } else if (handle == "/api/staff/car/info") {
            CarInfo(std::move(post), cgi);
        } else if (handle == "/api/staff/car/list") {
            CarList(std::move(post), cgi);
        } else if (handle == "/api/staff/authorize") {
            SendReply(NJson::JSON_NULL);
        } else if (handle == "/api/staff/commands") {
            AllowedCommands(*permissions);
        } else {
            SendReply(NJson::TMapBuilder("error", "invalid handler"), HTTP_BAD_REQUEST);
        }
    }

    void TReplier::CarControl(const NJson::TJsonValue& requestData, const NDrive::NPumpkin::TUserPermissions& permissions) const {
        auto command = NDrive::ParseCommand(requestData);
        TString imei;
        if (!NJson::ParseField(requestData["imei"], imei)) {
            SendReply(NJson::TMapBuilder("error", "cannot parse imei"), HTTP_BAD_REQUEST);
            return;
        }
        if (!command) {
            SendReply(NJson::TMapBuilder
                ("message", "cannot parse telematics command")
                ("request_body", requestData)
                ("error", command.GetError().what())
                , HTTP_BAD_REQUEST);
            return;
        }
        const auto& allowedCommands = permissions.AllowedCommands;
        if (auto it = std::find(allowedCommands.begin(), allowedCommands.end(), ToString(command->Code)); it == allowedCommands.end()) {
            SendReply(NJson::TMapBuilder
                ("error", "command is forbidden")
                ("code", ToString(command->Code))
                , HTTP_FORBIDDEN);
            return;
        }
        auto timeout = TDuration::Zero();
        if (!NJson::ParseField(requestData["timeout"], timeout)) {
            SendReply(NJson::TMapBuilder
                ("error", TStringBuilder() << "cannot parse timeout"), HTTP_BAD_REQUEST);
            return;
        }
        auto handler = Pumpkin.GetTelematicsClient().Command(imei, *command, timeout);
        handler.Subscribe([context = Context, cmd = *command](const NDrive::TTelematicsClient::THandler& handler) {
            auto status = handler.GetStatus();
            auto message = handler.GetMessage();
            NDrive::TEventLog::Log("FinishTelematicsCommand", NJson::TMapBuilder
                ("command", NJson::ToJson(cmd))
                ("id", handler.GetId())
                ("events", NJson::ToJson(handler.GetEvents()))
                ("shard", handler.GetShard())
                ("status", ToString(status))
                ("message", message)
                ("termination_processed", handler.GetTerminationProcessed())
            );
            NJson::TJsonValue resp = NJson::TMapBuilder("status", ToString(status))
                                                       ("message", message);
            if (auto response = handler.GetResponse<NDrive::TTelematicsClient::TGetParameterResponse>()) {
                resp["sensor"] = response->Sensor.FormatValue();
            }
            ::SendReply(context, std::move(resp), status == NDrive::TTelematicsClient::EStatus::Success ? HTTP_OK : HTTP_INTERNAL_SERVER_ERROR);
        });
    }

    void TReplier::CarSearch(const NJson::TJsonValue& requestData) const {
        TString prefix;
        if (!NJson::ParseField(requestData["prefix"], prefix)) {
            SendReply(NJson::TMapBuilder
                ("error", TStringBuilder() << "can't parse prefix"), HTTP_BAD_REQUEST);
            return;
        }
        ui32 limit = 10;
        if (!NJson::ParseField(requestData["limit"], limit)) {
            SendReply(NJson::TMapBuilder
                ("error", TStringBuilder() << "can't parse limit"), HTTP_BAD_REQUEST);
            return;
        }
        auto cars = Pumpkin.GetSimpleCarsDataStorage().FindCarsByPrefix(prefix, limit);
        SendReply(NJson::ToJson(cars));
    }

    void TReplier::CarInfo(const NJson::TJsonValue& requestData, const TCgiParameters& cgi) const {
        TString imei;
        if (!NJson::ParseField(requestData["imei"], imei)) {
            SendReply(NJson::TMapBuilder
                ("error", "cannot parse imei"), HTTP_BAD_REQUEST);
            return;
        }
        TString number;
        if (!NJson::ParseField(requestData["number"], number)) {
            SendReply(NJson::TMapBuilder
                ("error", "cannot parse number"), HTTP_BAD_REQUEST);
            return;
        }
        if (!imei && !number) {
            SendReply(NJson::TMapBuilder
                ("error", "imei or number field is expected"), HTTP_BAD_REQUEST);
            return;
        }
        TDuration timeout = Pumpkin.GetSensorsAPIQueryTimeout();
        auto timeoutStr = cgi.Get("timeout");
        if (timeoutStr) {
            if (ui64 value; TryFromString<ui64>(timeoutStr, value)) {
                timeout = TDuration::MicroSeconds(value);
            }
        }

        TMaybe<TSimpleCarData> carData;
        if (imei) {
            carData = Pumpkin.GetSimpleCarsDataStorage().GetCarByIMEI(imei);
        } else {
            carData = Pumpkin.GetSimpleCarsDataStorage().GetCarByNumber(number);
        }
        if (!carData) {
            SendReply(NJson::TMapBuilder
                ("error", "can't find car"), HTTP_NOT_FOUND);
            return;
        }
        TVector<NDrive::TSensorId> ids{
            CAN_FUEL_LEVEL_P,
            CAN_ODOMETER_KM,
            CAN_DRIVER_DOOR,
            CAN_PASS_DOOR,
            CAN_L_REAR_DOOR,
            CAN_R_REAR_DOOR,
            CAN_HOOD,
            CAN_TRUNK,
            CAN_ENGINE_IS_ON,
            VEGA_ACC_VOLTAGE,
            VEGA_POWER_VOLTAGE,
            VEGA_SPEED,
            VEGA_NRF_VISIBLE_MARKS_BF,
            VEGA_NRF_BATTLOW_MARKS_BF,
            VEGA_NRF_MARK_ID_1,
            VEGA_LAT,
            VEGA_LON,
            VEGA_DIR,
            NDrive::NVega::DigitalOutput<3>(),
            NDrive::NVega::SvrRawState,
        };
        auto sensorsF = Pumpkin.GetSensorAPI().GetSensors(carData->GetIMEI(), ids);
        auto imeiToLocations = Pumpkin.GetSensorsStorage().GetLocations({ carData->GetIMEI() });
        auto deadline = Context->GetRequestStartTime() + timeout;
        NThreading::Subscribe(sensorsF, [context = Context, carData = *carData, imeiToLocations](const NThreading::TFuture<NDrive::ISensorApi::TMultiSensor>& f) {
            try {
                auto sensors = f.GetValue();
                NJson::TJsonValue sensorsReport;
                for (const auto& sensor : sensors) {
                    sensorsReport.AppendValue(sensor.GetJsonReport());
                }
                NJson::TJsonValue res = NJson::ToJson(carData);
                res["sensors"] = std::move(sensorsReport);
                if (auto it = imeiToLocations.find(carData.GetIMEI()); it != imeiToLocations.end()) {
                    for (const auto& [_, location] : it->second) {
                        res["locations"].AppendValue(location.ToJson());
                    }
                }
                ::SendReply(context, std::move(res), HTTP_OK);
            } catch (const std::exception& e) {
                ::SendReply(context, NJson::TMapBuilder("error", FormatExc(e)), HTTP_INTERNAL_SERVER_ERROR);
            }
        }, deadline);
    }

    void TReplier::CarList(const NJson::TJsonValue& /* requestData */, const TCgiParameters& /* cgi */) const {
        auto imeis = Pumpkin.GetSimpleCarsDataStorage().GetCarsIMEIs();
        auto imeiToLocations = Pumpkin.GetSensorsStorage().GetLocations(imeis);
        const auto& carsData = Pumpkin.GetSimpleCarsDataStorage().GetCarsData();

        NJson::TJsonValue report;
        for (const auto& carData : carsData) {
            NJson::TJsonValue carReport = NJson::ToJson(carData);
            if (auto it = imeiToLocations.find(carData.GetIMEI()); it != imeiToLocations.end()) {
                for (const auto& [_, location] : it->second) {
                    carReport["locations"].AppendValue(location.ToJson());
                }
            }
            report.AppendValue(std::move(carReport));
        }
        SendReply(std::move(report));
    }

    void TReplier::AllowedCommands(const NDrive::NPumpkin::TUserPermissions& permissions) const {
        SendReply(NJson::TMapBuilder
            ("allowed_commands", NJson::ToJson(permissions.AllowedCommands)));
    }

    void TReplier::SendReply(NJson::TJsonValue&& reply, ui32 code) const {
        ::SendReply(Context, std::move(reply), code);
    }

    NJson::TJsonValue TReplier::GetPost() const {
        const TBlob& buf = Context->GetBuf();
        const TStringBuf str(buf.AsCharPtr(), buf.Size());
        NJson::TJsonValue result;
        if (!NJson::ReadJsonFastTree(str, &result)) {
            throw TCodedException(HTTP_BAD_REQUEST) << "cannot parse POST data as json";
        }
        return result;
    }
}
