#include "sensors_storage.h"

#include <drive/telematics/protocol/vega.h>
#include <drive/telematics/server/location/names.h>

namespace NDrive::NPumpkin {
    TVector<TString> TSensorsStorage::LocationsNames = {
        NDrive::RawLocationName,
        NDrive::LBSLocationName,
        NDrive::LinkedLocationName,
        NDrive::HeadLocationName,
        NDrive::BeaconsLocationName,
        NDrive::GeocodedLocationName
    };

    TSensorsStorage::TSensorsStorage(TAtomicSharedPtr<NDrive::ISensorApi> sensorAPI, TDuration updateInterval)
        : IAutoActualization("TSensorsStorage", updateInterval)
        , SensorAPI(std::move(sensorAPI))
    {
        if (!Start()) {
            ythrow yexception() << "can't start " << GetName();
        }
    }

    TSensorsStorage::~TSensorsStorage() {
        if (!Stop()) {
            ERROR_LOG << "can't stop " << GetName();
        }
    }

    bool TSensorsStorage::Refresh() {
        TMap<TString, NThreading::TFuture<NDrive::ISensorApi::TLocations>> locationFutures;
        for (const auto& name : LocationsNames) {
            locationFutures[name] = SensorAPI->GetLocations(name);
        }
        TMap<TString, NDrive::ISensorApi::TLocations> locations;
        for (auto& [name, locationF] : locationFutures) {
            locations[name] = locationFutures[name].ExtractValueSync();
        }
        TWriteGuard guard(Mutex);
        Locations = std::move(locations);
        return true;
    }

    TMap<TString, TMap<TString, NDrive::TLocation>> TSensorsStorage::GetLocations(const TVector<TString>& imeis) const {
        TReadGuard guard(Mutex);
        TMap<TString, TMap<TString, NDrive::TLocation>> imeiToLocations;
        for (const auto& imei : imeis) {
            for (const auto& locationName: LocationsNames) {
                if (auto it = Locations.at(locationName).find(imei); it != Locations.at(locationName).end()) {
                    imeiToLocations[imei][locationName] = it->second;
                }
            }
        }
        return imeiToLocations;
    }
}
