#include <drive/pumpkin/library/json_text_patcher.h>
#include <drive/pumpkin/library/logging.h>
#include <drive/pumpkin/library/pumpkin.h>

#include <kernel/daemon/common/environment.h>
#include <kernel/daemon/signal_handler.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/digest/murmur.h>
#include <util/folder/path.h>
#include <util/stream/file.h>
#include <util/system/execpath.h>
#include <util/system/hostname.h>
#include <util/system/info.h>
#include <util/string/subst.h>

void AddEnvironmentVariables(NDrive::NPumpkin::TLuaJsonTextPatcher& patcher) {
    patcher.AddVariable("NCPU", ToString(NSystemInfo::CachedNumberOfCpus()));
    patcher.AddVariable("PAGESIZE", ToString(NSystemInfo::GetPageSize()));
    patcher.AddVariable("_BIN_DIRECTORY", TFsPath(GetExecPath()).Dirname());
    const TString& host = HostName();
    patcher.AddVariable("HOSTNAME", host);
    patcher.AddVariable("HOSTNAME_HASH", ToString(MurmurHash<ui32>(host.data(), host.size())));

    for (const auto& [key, value] : NUtil::GetEnvironmentVariables()) {
        patcher.AddVariable(key, value);
    }
}

int main(int argc, const char** argv) {
    NLastGetopt::TOpts options;
    NDrive::NPumpkin::TLuaJsonTextPatcher patcher;
    options.AddCharOption('V', "set config Lua variable key to value").RequiredArgument("key=value")
        .KVHandler([&patcher](TString key, TString value){ patcher.AddVariable(key, value); });
    options.SetFreeArgsMin(1);
    options.SetFreeArgTitle(0, "CONFIG_FILE", "path to pumpkin configuration file");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString configFile = res.GetFreeArgs().at(0);

    AddEnvironmentVariables(patcher);
    const NDrive::NPumpkin::TConfig config{TFsPath(configFile), &patcher};
    NDrive::NPumpkin::InitLogging(config);
    NDrive::TPumpkin::SetConfig(config);
    Singleton<NDrive::TPumpkin>()->Run();
    Singleton<NDrive::TPumpkin>()->Wait();
    return 0;
}
