#include <drive/pumpkin/ut/library/helper.h>
#include <drive/telematics/server/location/beacon_recognizer.h>
#include <drive/telematics/server/location/locator.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/location/names.h>
#include <drive/telematics/server/ut/library/helper.h>
#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(TelematicsPumpkin) {
    Y_UNIT_TEST(TestCarControl) {
        TTelematicServerBuilder tmBuilder;


        NDrive::NPumpkin::TConfig config(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_config.json"));
        config.MutableTelematicsClientConfig()->MutableTelematicsClientOptions()->AddShards("localhost:" + ToString(tmBuilder.GetHttpPort()));
        config.MutableSimpleCarsDataStorageConfig()->SetCarsDataFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/cars_data.json"));
        config.MutableUsersPermissionsStorageConfig()->SetUsersPermissionsFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_users_permissions.json"));

        NDrive::TPumpkinBuilder tpBuilder;
        tpBuilder.Run(config);

        tmBuilder.Run();
        auto context = MakeHolder<NDrive::TTelematicsClientContext>();
        int mnc = 1212;
        context->SetMNC(mnc);
        TString imei = "98348934789";
        auto emulator = tmBuilder.BuildEmulator(imei, std::move(context));

        NJson::TJsonValue request = NJson::TMapBuilder
                                            ("imei", imei)
                                            ("command", "GET_PARAM")
                                            ("id", VEGA_MNC);

        auto response = tpBuilder.CarControl(request);
        UNIT_ASSERT_VALUES_EQUAL(response.first, HTTP_OK);
        UNIT_ASSERT_VALUES_EQUAL(response.second["sensor"].GetInteger(), mnc);
        tpBuilder.Stop();
    }

    Y_UNIT_TEST(TestCarControlNotPermitted) {
        TTelematicServerBuilder tmBuilder;


        NDrive::NPumpkin::TConfig config(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_config.json"));
        config.MutableTelematicsClientConfig()->MutableTelematicsClientOptions()->AddShards("localhost:" + ToString(tmBuilder.GetHttpPort()));
        config.MutableSimpleCarsDataStorageConfig()->SetCarsDataFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/cars_data.json"));
        config.MutableUsersPermissionsStorageConfig()->SetUsersPermissionsFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_users_permissions.json"));

        NDrive::TPumpkinBuilder tpBuilder;
        tpBuilder.Run(config);

        tmBuilder.Run();
        auto context = MakeHolder<NDrive::TTelematicsClientContext>();
        int mnc = 1212;
        context->SetMNC(mnc);
        TString imei = "98348934789";
        auto emulator = tmBuilder.BuildEmulator(imei, std::move(context));

        NJson::TJsonValue request = NJson::TMapBuilder
                                            ("imei", imei)
                                            ("command", "SET_PARAM")
                                            ("id", VEGA_MNC)
                                            ("value", 99);

        auto response = tpBuilder.CarControl(request);
        UNIT_ASSERT_VALUES_EQUAL(response.first, HTTP_FORBIDDEN);
        tpBuilder.Stop();
    }

    Y_UNIT_TEST(TestCarSearch) {
        NDrive::NPumpkin::TConfig config(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_config.json"));
        config.MutableSimpleCarsDataStorageConfig()->SetCarsDataFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/cars_data.json"));
        config.MutableUsersPermissionsStorageConfig()->SetUsersPermissionsFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_users_permissions.json"));
        NDrive::TPumpkinBuilder tpBuilder;
        tpBuilder.Run(config);
        NJson::TJsonValue request = NJson::TMapBuilder("prefix", "abcd");
        auto response = tpBuilder.CarSearch(request);
        UNIT_ASSERT_VALUES_EQUAL(response.first, HTTP_OK);
        UNIT_ASSERT(response.second.IsArray());
        const auto& carList = response.second.GetArray();
        UNIT_ASSERT_VALUES_EQUAL(carList.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(carList[0].GetStringRobust(), NJson::TJsonValue(NJson::TMapBuilder("id", "abcd-dcba")("imei", "111111")("number", "tramvai")).GetStringRobust());
        tpBuilder.Stop();
    }

    Y_UNIT_TEST(TestCarsStorage) {
        {
            TVector<NDrive::NPumpkin::TSimpleCarData> carsData;
            carsData.emplace_back().SetId("id1").SetIMEI("imei1").SetNumber("number1");
            carsData.emplace_back().SetId("id2").SetIMEI("imei2").SetNumber("number2");
            auto secondCar = carsData[1];
            NDrive::NPumpkin::TSimpleCarsDataStorage storage(std::move(carsData));
            auto cars = storage.FindCarsByPrefix("id2", 1);
            UNIT_ASSERT_VALUES_EQUAL(cars.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(cars.begin()->GetId(), secondCar.GetId());
            UNIT_ASSERT_VALUES_EQUAL(cars.begin()->GetIMEI(), secondCar.GetIMEI());
            UNIT_ASSERT_VALUES_EQUAL(cars.begin()->GetNumber(), secondCar.GetNumber());
        }
        {
            TVector<NDrive::NPumpkin::TSimpleCarData> carsData;
            carsData.emplace_back().SetId("a").SetIMEI("b").SetNumber("ca");
            carsData.emplace_back().SetId("b").SetIMEI("a").SetNumber("c");
            auto firstCar = carsData[0];
            auto secondCar = carsData[1];
            NDrive::NPumpkin::TSimpleCarsDataStorage storage(std::move(carsData));
            auto cars = storage.FindCarsByPrefix("a", 2);
            UNIT_ASSERT_VALUES_EQUAL(cars.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(cars.begin()->GetId(), firstCar.GetId());
            UNIT_ASSERT_VALUES_EQUAL(cars.begin()->GetIMEI(), firstCar.GetIMEI());
            UNIT_ASSERT_VALUES_EQUAL(cars.begin()->GetNumber(), firstCar.GetNumber());
            UNIT_ASSERT_VALUES_EQUAL(cars.rbegin()->GetId(), secondCar.GetId());
            UNIT_ASSERT_VALUES_EQUAL(cars.rbegin()->GetIMEI(), secondCar.GetIMEI());
            UNIT_ASSERT_VALUES_EQUAL(cars.rbegin()->GetNumber(), secondCar.GetNumber());
        }
        {
            TVector<NDrive::NPumpkin::TSimpleCarData> carsData;
            carsData.emplace_back().SetId("a").SetIMEI("b").SetNumber("ca");
            carsData.emplace_back().SetId("b").SetIMEI("a").SetNumber("c");
            NDrive::NPumpkin::TSimpleCarsDataStorage storage(std::move(carsData));
            auto cars = storage.FindCarsByPrefix("a", 1);
            UNIT_ASSERT_VALUES_EQUAL(cars.size(), 1);
        }
        {
            TVector<NDrive::NPumpkin::TSimpleCarData> carsData;
            carsData.emplace_back().SetId("a").SetIMEI("x");
            carsData.emplace_back().SetId("b").SetIMEI("xx");
            carsData.emplace_back().SetId("c").SetIMEI("xx");
            carsData.emplace_back().SetId("d").SetIMEI("xx");
            carsData.emplace_back().SetId("e").SetIMEI("xx");
            carsData.emplace_back().SetId("f").SetIMEI("xx");
            carsData.emplace_back().SetId("g").SetIMEI("xx");
            carsData.emplace_back().SetId("h").SetIMEI("xxx");
            NDrive::NPumpkin::TSimpleCarsDataStorage storage(std::move(carsData));
            auto cars = storage.FindCarsByPrefix("x", 10);
            UNIT_ASSERT_VALUES_EQUAL(cars.size(), 8);
            cars = storage.FindCarsByPrefix("z", 100);
            UNIT_ASSERT_VALUES_EQUAL(cars.size(), 0);
        }
    }
    Y_UNIT_TEST(TestCarInfo) {
        TTelematicServerBuilder tmBuilder;

        NDrive::NPumpkin::TConfig config(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_config.json"));
        config.MutableTelematicsClientConfig()->MutableTelematicsClientOptions()->AddShards("localhost:" + ToString(tmBuilder.GetHttpPort()));
        config.MutableSimpleCarsDataStorageConfig()->SetCarsDataFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/cars_data.json"));
        config.MutableUsersPermissionsStorageConfig()->SetUsersPermissionsFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_users_permissions.json"));

        NDrive::TPumpkinBuilder tpBuilder;
        tpBuilder.Run(config);

        tmBuilder.Run();
        const auto &api = tmBuilder.GetServer()->GetSensors();

        auto context = MakeHolder<NDrive::TTelematicsClientContext>();
        TGeoCoord coord{12.58643489401084, 55.73761541977004};
        context->SetCurrentPosition(coord);
        TString imei = "111111";
        auto emulator = tmBuilder.BuildEmulator(imei, std::move(context));
        for (int i = 0; i < 5; ++i) {
            auto sensors = api->GetSensors({ imei }, { VEGA_LAT, VEGA_LON });
            UNIT_ASSERT(sensors.HasValue());
            if (sensors.GetValue().size() != 2) {
                Sleep(TDuration::Seconds(2));
                continue;
            }
            for (const auto& s : sensors.GetValue()) {
                if (s.Id == VEGA_LAT) {
                    UNIT_ASSERT_DOUBLES_EQUAL(std::get<double>(s.Value), 55.73761541977004, 1e-4);
                } else {
                    UNIT_ASSERT_DOUBLES_EQUAL(std::get<double>(s.Value), 12.58643489401084, 1e-4);
                }
            }
            break;
        }
        for (auto request : { NJson::TJsonValue(NJson::TMapBuilder("number", "tramvai")), NJson::TJsonValue(NJson::TMapBuilder("imei", imei)) }) {
            auto response = tpBuilder.CarInfo(request);
            UNIT_ASSERT_VALUES_EQUAL(response.first, HTTP_OK);
            UNIT_ASSERT(response.second.IsMap());
            UNIT_ASSERT_VALUES_EQUAL(response.second["id"].GetString(), "abcd-dcba");
            UNIT_ASSERT_VALUES_EQUAL(response.second["number"].GetString(), "tramvai");
            UNIT_ASSERT_VALUES_EQUAL(response.second["imei"].GetString(), imei);
            const auto& sensorsResp = response.second["sensors"];
            UNIT_ASSERT(sensorsResp.IsArray());
            for (const auto& sensorData : sensorsResp.GetArray()) {
                if (sensorData["name"].GetString() == "VEGA_LON") {
                    UNIT_ASSERT_DOUBLES_EQUAL(sensorData["value"].GetDouble(), 12.58643489401084, 1e-4);
                } else if (sensorData["name"].GetString() == "VEGA_LAT") {
                    UNIT_ASSERT_DOUBLES_EQUAL(sensorData["value"].GetDouble(), 55.73761541977004, 1e-4);
                }
            }
        }
        tpBuilder.Stop();
        // NJson::TJsonValue request;
        // auto response = tpBuilder.CarInfo(request);
        // UNIT_ASSERT_VALUES_EQUAL(response.first, HTTP_NOT_FOUND);
    }

    Y_UNIT_TEST(TestCarList) {
        TTelematicServerBuilder tmBuilder;

        NDrive::NPumpkin::TConfig config(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_config.json"));
        config.MutableTelematicsClientConfig()->MutableTelematicsClientOptions()->AddShards("localhost:" + ToString(tmBuilder.GetHttpPort()));
        config.MutableSimpleCarsDataStorageConfig()->SetCarsDataFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/cars_data.json"));
        config.MutableUsersPermissionsStorageConfig()->SetUsersPermissionsFile(JoinFsPaths(ArcadiaSourceRoot(), "drive/pumpkin/ut/test_users_permissions.json"));

        NDrive::TPumpkinBuilder tpBuilder;
        tpBuilder.Run(config);

        NDrive::NVega::TBeaconInfos fakeBeaconInfos;
        for (size_t i = 0; i < fakeBeaconInfos.Elements.size(); ++i) {
            fakeBeaconInfos.Elements[i].MutableRawUUID() = {0, 0, 0, 0};
            fakeBeaconInfos.Elements[i].MutableRawMajor() = 0;
            fakeBeaconInfos.Elements[i].MutableRawMinor() = 0;
        }
        TMap<TString, NDrive::IBeaconRecognizer::TLocationData> keyToLocationData;
        keyToLocationData[NDrive::IBeaconRecognizer::ConvertToKey(fakeBeaconInfos.Elements[0])] = NDrive::IBeaconRecognizer::TLocationData{"my_location", 45., 23.};
        auto beaconRecognizer = MakeHolder<NDrive::TStaticBeaconRecognizer>(std::move(keyToLocationData));
        auto tmConfig = tmBuilder.GetConfig();
        tmConfig->SetBeaconsRefreshInterval(TDuration::Zero());
        auto server = tmBuilder.GetServer();
        server->GetLocator().SetBeaconRecognizer(std::move(beaconRecognizer));
        tmBuilder.Run();

        const TString shard = "localhost:" + ToString(tmConfig->GetClientServerOptions().Port);
        const auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient::TOptions options;
        options.DefaultWaitConnectionTimeout = TDuration::Seconds(20);
        NDrive::TTelematicsClient client(sd, options);

        const auto &api = tmBuilder.GetServer()->GetSensors();

        TVector<TString> imeis{"111111", "000723414"};
        TVector<TAtomicSharedPtr<NDrive::TCarEmulator>> emulators;
        for (int i = 0; i < 2; ++i) {
            auto context = MakeHolder<NDrive::TTelematicsClientContext>();
            TGeoCoord coord{12.58643489401084, 55.73761541977004};
            context->SetCurrentPosition(coord);
            if (i == 0) {
                constexpr int beaconSensorsNumber = 10;
                TVector<NDrive::NVega::TBeaconInfos> beaconInfos(beaconSensorsNumber, fakeBeaconInfos);
                context->SetBeaconsInfos(std::move(beaconInfos));
            }
            emulators.emplace_back(tmBuilder.BuildEmulator(imeis[i], std::move(context)));
        }

        {
            NDrive::NVega::TCommand command(NDrive::NVega::ECommandCode::SCENARIO_QUERY_BEACONS);
            auto handle = client.Command(imeis[0], command);
            handle.GetFuture().Wait();
            auto response = handle.GetResponse<NDrive::TTelematicsClient::TCommonResponse>();
            UNIT_ASSERT(response);
            UNIT_ASSERT(response->GetStatus() == NDrive::TTelematicsClient::EStatus::Success);
        }
        bool pushedBeaconsLocation = false;
        for (int i = 0; i < 5; ++i) {
            auto locationsF = api->GetLocations({ imeis[0] }, NDrive::BeaconsLocationName);
            UNIT_ASSERT(locationsF.HasValue());
            const auto& locations = locationsF.GetValue();
            if (locations.size() != 1) {
                Sleep(TDuration::Seconds(1));
                continue;
            }

            pushedBeaconsLocation = true;
            break;
        }
        UNIT_ASSERT(pushedBeaconsLocation);
        Sleep(TDuration::Seconds(config.GetSensorsStorageConfig().GetUpdateIntervalSeconds()));

        auto response = tpBuilder.CarList(NJson::JSON_NULL);
        UNIT_ASSERT_VALUES_EQUAL(response.first, HTTP_OK);
        UNIT_ASSERT(response.second.IsArray());

        const auto& cars = response.second.GetArray();
        UNIT_ASSERT_VALUES_EQUAL(cars.size(), 2);
        for (const auto& car : cars) {
            bool goodCar = car["id"].GetString() == "abcd-dcba" || car["id"].GetString() == "hengen-dfdfaw";
            UNIT_ASSERT(goodCar);
            if (car["id"].GetString() == "abcd-dcba") {
                UNIT_ASSERT_VALUES_EQUAL(car["imei"].GetString(), "111111");
                UNIT_ASSERT_VALUES_EQUAL(car["number"].GetString(), "tramvai");
                bool foundBeaconsLocation = false;
                for (const auto& locationsData : car["locations"].GetArray()) {
                    if (locationsData["name"].GetString() == NDrive::BeaconsLocationName) {
                        foundBeaconsLocation = true;
                        auto content = locationsData["content"].GetString();
                        UNIT_ASSERT_VALUES_EQUAL(content, "my_location");
                    }
                }
                UNIT_ASSERT(foundBeaconsLocation);
            } else if (car["id"].GetString() == "hengen-dfdfaw") {
                UNIT_ASSERT_VALUES_EQUAL(car["imei"].GetString(), "000723414");
                UNIT_ASSERT_VALUES_EQUAL(car["number"].GetString(), "trolleybus");
            }
        }
        tpBuilder.Stop();
    }
}
