from cars.admin.serializers.user_simple import SimpleUserSerializer
from rest_framework.serializers import (
    CharField, ModelSerializer,
    SerializerMethodField, UUIDField,
)

from cars.carsharing.models import CarRegistryDocument
from cars.carsharing.models.car_document import CarDocument, CarDocumentAssignment
from cars.carsharing.models.car_hardware import CarHardwareVega, CarHardwareModem, CarHardwareHead, CarHardwareBeacon, \
    CarHardwareSim
from cars.django.serializers import BaseSerializer, TimestampField
from .carsharing import CarsharingInsuranceSerializer
from .carsharing_simple import SimpleCarSerializer


class CarDocumentsListViewArgumentsSerializer(BaseSerializer):

    type = CharField(required=False)


class CarDocumentAssignmentsListViewArgumentsSerializer(BaseSerializer):

    type = CharField(required=False)
    car_id = UUIDField(required=False)
    assigned_by = UUIDField(required=False)
    unassigned_by = UUIDField(required=False)


class CarHardwareSimSerializer(ModelSerializer):

    icc = SerializerMethodField()

    class Meta:
        model = CarHardwareSim
        fields = '__all__'

    def get_icc(self, sim):
        return str(sim.icc)


class CarHardwareBeaconSerializer(ModelSerializer):

    sim = CarHardwareSimSerializer()

    class Meta:
        model = CarHardwareBeacon
        fields = '__all__'


class CarHardwareHeadSerializer(ModelSerializer):

    class Meta:
        model = CarHardwareHead
        fields = '__all__'


class CarHardwareModemSerializer(ModelSerializer):

    sim = CarHardwareSimSerializer()

    class Meta:
        model = CarHardwareModem
        fields = '__all__'


class CarHardwareVegaSerializer(ModelSerializer):

    primary_sim = CarHardwareSimSerializer()
    secondary_sim = CarHardwareSimSerializer()

    class Meta:
        model = CarHardwareVega
        fields = '__all__'


class CarRegistryDocumentSerializer(ModelSerializer):

    class Meta:
        model = CarRegistryDocument
        fields = '__all__'


class CarDocumentSerializer(ModelSerializer):

    added_at = TimestampField()
    added_by = SimpleUserSerializer()
    impl = SerializerMethodField()

    class Meta:
        model = CarDocument
        fields = [
            'id',
            'type',
            'added_at',
            'added_by',
            'impl',
        ]

    def get_impl(self, obj):
        impl = obj.get_impl()
        if obj.get_type() is CarDocument.Type.CAR_HARDWARE_MODEM:
            return CarHardwareModemSerializer(impl).data
        elif obj.get_type() is CarDocument.Type.CAR_HARDWARE_HEAD:
            return CarHardwareHeadSerializer(impl).data
        elif obj.get_type() is CarDocument.Type.CAR_HARDWARE_BEACON:
            return CarHardwareBeaconSerializer(impl).data
        elif obj.get_type() is CarDocument.Type.CAR_HARDWARE_VEGA:
            return CarHardwareVegaSerializer(impl).data
        elif obj.get_type() is CarDocument.Type.CAR_HARDWARE_SIM:
            return CarHardwareSimSerializer(impl).data
        elif obj.get_type() is CarDocument.Type.CAR_REGISTRY_DOCUMENT:
            return CarRegistryDocumentSerializer(impl).data
        elif obj.get_type() is CarDocument.Type.CAR_INSURANCE_POLICY:
            return CarsharingInsuranceSerializer(impl).data
        else:
            raise RuntimeError('unreachable')


class CarDocumentAssignmentSerializer(ModelSerializer):

    document = CarDocumentSerializer()
    car = SimpleCarSerializer()
    assigned_at = TimestampField()
    assigned_by = SimpleUserSerializer()
    unassigned_at = TimestampField()
    unassigned_by = SimpleUserSerializer()

    class Meta:
        model = CarDocumentAssignment
        fields = [
            'id',
            'document',
            'car',
            'assigned_at',
            'assigned_by',
            'unassigned_at',
            'unassigned_by',
        ]
