from rest_framework.serializers import (
    CharField,
    DecimalField,
    ModelSerializer,
    SerializerMethodField,
)

from cars.carsharing.models import Car, CarInsurance
from cars.django.serializers import BaseSerializer
from cars.orders.models.order import Order
from ..serializers.order import OrderSerializer
from .carsharing_simple import (
    CarLocationSerializer, CarModelSerializer, CarTelematicsStateSerializer,
)


class CarListViewArgumentsSerializer(BaseSerializer):

    q = CharField(required=False)


class UpdateFuelCardArgumentsSerializer(BaseSerializer):

    number = CharField(required=True)


class CarsharingInsuranceSerializer(ModelSerializer):

    base_cost = DecimalField(max_digits=9, decimal_places=2)
    per_minute_cost = DecimalField(max_digits=9, decimal_places=2)

    class Meta:
        model = CarInsurance
        fields = [
            'agreement_number',
            'agreement_partner_number',
            'base_cost',
            'per_minute_cost',
        ]


class CarSerializer(ModelSerializer):

    location = CarLocationSerializer()
    model = CarModelSerializer()
    telematics_state = CarTelematicsStateSerializer()
    order = SerializerMethodField()
    insurance = CarsharingInsuranceSerializer()

    class Meta:
        model = Car
        fields = [
            'id',
            'imei',
            'vin',
            'contract_id',
            'registration_id',
            'registration_date',
            'number',
            'location',
            'model',
            'telematics_state',
            'status',
            'order',
            'insurance',
            'fuel_card_number',
            'responsible_picker',
        ]

    def get_order(self, obj):
        order = (
            Order.objects
            .select_related('user__registration_state')
            .prefetch_related('items')
            .filter(
                completed_at__isnull=True,
                items__carsharing_reservation__car_id=obj.id,
            )
            .first()
        )

        if order:
            order_data = OrderSerializer(order, context=self.context).data
        else:
            order_data = None

        return order_data
