from rest_framework.serializers import CharField, ModelSerializer, SerializerMethodField

from cars.carsharing.models.tariff_plan import CarsharingTariffPlan
from cars.carsharing.models.tariff_plan_entry import CarsharingTariffPlanEntry
from cars.django.serializers import BaseSerializer, TimestampField
from .user_simple import SimpleUserSerializer


class CarsharingTariffPlanEntrySerializer(ModelSerializer):

    class Meta:
        model = CarsharingTariffPlanEntry
        fields = [
            'id',
            'day_of_week',
            'start_time',
            'end_time',
            'ride_cost_per_minute',
            'parking_cost_per_minute',
        ]

    def to_internal_value(self, data):
        if data['start_time'] in ('00:00', '00:00:00'):
            data['start_time'] = None

        if data['end_time'] in ('24:00', '24:00:00'):
            data['end_time'] = None

        data = super().to_internal_value(data)

        return self.Meta.model(
            start_time=data['start_time'],
            end_time=data['end_time'],
            day_of_week=data.get('day_of_week'),
            ride_cost_per_minute=data['ride_cost_per_minute'],
            parking_cost_per_minute=data['parking_cost_per_minute'],
        )

    def to_representation(self, obj):
        data = super().to_representation(obj)

        if data['start_time'] is None:
            data['start_time'] = '00:00:00'

        if data['end_time'] is None:
            data['end_time'] = '24:00:00'

        return data


class CarsharingTariffPlanArgumentsSerializer(BaseSerializer):
    name = CharField()
    entries = CarsharingTariffPlanEntrySerializer(many=True)
    car_model_code = CharField(required=False)
    user_tag = CharField(required=False)


class CarsharingTariffPlanSerializer(ModelSerializer):
    created_by = SimpleUserSerializer()
    created_at = TimestampField()
    entries = CarsharingTariffPlanEntrySerializer(many=True)
    car_model = SerializerMethodField()
    timezone = SerializerMethodField()

    class Meta:
        model = CarsharingTariffPlan
        fields = [
            'id',
            'created_by',
            'created_at',
            'name',
            'entries',
            'car_model',
            'user_tag',
            'timezone',
        ]

    def get_car_model(self, obj):
        return {
            'code': obj.car_model_id,
        }

    def get_timezone(self, obj):
        return obj.get_timezone().zone
