from rest_framework.serializers import ModelSerializer, SerializerMethodField, UUIDField

from cars.django.serializers import BaseSerializer, TimestampField
from cars.eka.models import FuelCardActivation


class FuelCardActivationUsagesViewArgumentsSerializer(BaseSerializer):

    user_id = UUIDField(required=False)
    car_id = UUIDField(required=False)
    order_id = UUIDField(required=False)


class FuelCardActivationSerializer(ModelSerializer):

    activated_at = TimestampField()
    activated_via = SerializerMethodField()
    blocked_at = TimestampField()
    blocked_via = SerializerMethodField()

    bonus_added_at = SerializerMethodField()
    bonus_amount = SerializerMethodField()

    class Meta:
        model = FuelCardActivation
        fields = [
            'id',
            'order_id',
            'car_id',
            'activated_at',
            'activated_by',
            'activated_via',
            'blocked_at',
            'blocked_by',
            'blocked_via',
            'initial_fuel_level',
            'eventual_fuel_level',
            'bonus_added_at',
            'bonus_amount',
        ]

    def get_bonus_amount(self, obj):
        if obj.bonus_account_operation is None:
            return None
        return obj.bonus_account_operation.amount

    def get_bonus_added_at(self, obj):
        if obj.bonus_account_operation is None:
            return None
        return obj.bonus_account_operation.created_at.timestamp()

    def get_activated_via(self, obj):
        return self._translate_source(obj.activated_via)

    def get_blocked_via(self, obj):
        return self._translate_source(obj.activated_via)

    def _translate_source(self, source):
        if source == FuelCardActivation.Source.INTERNAL_CC.value:
            return 'Внутренний КЦ'
        elif source == FuelCardActivation.Source.EXTERNAL_CC.value:
            return 'Внешний КЦ'
        elif source == FuelCardActivation.Source.ROBOT_AUTO.value:
            return 'Робот по таймауту'
        elif source == FuelCardActivation.Source.ADMIN_PORTAL.value:
            return 'ЦУП'
        elif source == FuelCardActivation.Source.MOBILE_APP.value:
            return 'Мобильное приложение'
        else:
            return None
