import logging

from rest_framework.serializers import (
    NullBooleanField, CharField, ModelSerializer, SerializerMethodField, UUIDField,
)

import cars.settings
from cars.billing.models.card_payment import CardPayment
from cars.django.serializers import BaseSerializer, TimestampField
from cars.carsharing.models.acceptance import CarsharingAcceptance, CarsharingAcceptancePhoto
from cars.carsharing.models.parking import CarsharingParking
from cars.carsharing.models.reservation import CarsharingReservation
from cars.carsharing.models.ride import CarsharingRide
from cars.orders.models.order import Order
from cars.orders.models.order_item import OrderItem
from cars.orders.models.order_item_payment import OrderItemPayment
from cars.orders.models.order_item_tariff import (
    FixOrderItemTariffParams, OrderItemTariff, PerMinuteOrderItemTariffParams,
)
from .carsharing_simple import SimpleCarSerializer
from .user_simple import SimpleUserSerializer


LOGGER = logging.getLogger(__name__)


class OrderListArgumentsSerializer(BaseSerializer):
    q = CharField(required=False)
    user_id = UUIDField(required=False)
    car_id = UUIDField(required=False)

    is_completed = NullBooleanField(required=False)
    created_at_gte = TimestampField(required=False)
    created_at_lte = TimestampField(required=False)
    completed_at_gte = TimestampField(required=False)
    completed_at_lte = TimestampField(required=False)


class CarAllInfoByIMEIArgumentsSerializer(BaseSerializer):

    timestamp = TimestampField(required=True)
    imei = CharField(required=True)


class CarAllInfoByIMEIResponseSerializer(BaseSerializer):

    action_id = SerializerMethodField()
    car_id = SerializerMethodField()
    session_id = SerializerMethodField()
    user_id = SerializerMethodField()
    car_status = SerializerMethodField()
    timestamp_start = SerializerMethodField()
    timestamp_finish = SerializerMethodField()

    def get_action_id(self, obj):
        if obj is not None:
            action_id = obj.id
        else:
            action_id = None
        return action_id

    def get_car_id(self, obj):
        return obj.get_impl().car_id

    def get_session_id(self, obj):
        return obj.order_id

    def get_user_id(self, obj):
        return obj.order.user_id

    def get_car_status(self, obj):
        return self.context.get('status') or obj.type[11:]  # remove 'carsharing_' prefix

    def get_timestamp_start(self, obj):
        event_time = (self.context.get('timestamp_start') or obj.started_at)
        if event_time:
            timestamp_start = event_time.timestamp()
        else:
            timestamp_start = None
        return timestamp_start

    def get_timestamp_finish(self, obj):
        event_time = (self.context.get('timestamp_finish') or obj.finished_at)
        if event_time:
            timestamp_finish = event_time.timestamp()
        else:
            timestamp_finish = None
        return timestamp_finish


class CarsharingAcceptancePhotoSerializer(BaseSerializer):
    id = CharField()
    mime_type = CharField()
    link = CharField()
    submitted_at = TimestampField()
    order_id = SerializerMethodField()

    class Meta:
        model = CarsharingAcceptancePhoto
        fields = [
            'id',
            'link',
            'mime_type',
            'submitted_at',
            'order_id'
        ]

    def get_order_id(self, obj):
        return obj.acceptance.order_item.order_id


class CarsharingAcceptanceSerializer(BaseSerializer):
    car = SimpleCarSerializer()
    photos = CarsharingAcceptancePhotoSerializer(many=True)
    car_condition = CharField()
    comment = CharField()
    insurance_present = NullBooleanField()
    fuel_card_present = NullBooleanField()
    sts_present = NullBooleanField()

    class Meta:
        model = CarsharingAcceptance
        fields = [
            'car',
            'car_condition',
            'comment',
            'fuel_card_present',
            'insurance_present',
            'sts_present',
            'photos',
        ]


class CarsharingReservationSerializer(BaseSerializer):
    car = SimpleCarSerializer()

    class Meta:
        model = CarsharingReservation
        fields = [
            'car'
        ]


class CarsharingRideSerializer(ModelSerializer):
    car = SimpleCarSerializer()

    class Meta:
        model = CarsharingRide
        fields = [
            'car',
            'mileage'
        ]


class CarsharingParkingSerializer(BaseSerializer):
    car = SimpleCarSerializer()

    class Meta:
        model = CarsharingParking
        fields = [
            'car'
        ]


class FixOrderItemTariffParamsSerializer(ModelSerializer):

    class Meta:
        model = FixOrderItemTariffParams
        fields = [
            'cost',
        ]


class PerMinuteOrderItemTariffParamsSerializer(ModelSerializer):

    class Meta:
        model = PerMinuteOrderItemTariffParams
        fields = [
            'cost_per_minute',
        ]


class OrderItemTariffSerializer(ModelSerializer):

    params = SerializerMethodField()

    class Meta:
        model = OrderItemTariff
        fields = [
            'type',
            'params',
        ]

    def get_params(self, obj):
        type_ = obj.get_type()
        params = obj.get_params()
        if type_ is OrderItemTariff.Type.FIX:
            params_data = FixOrderItemTariffParamsSerializer(params).data
        elif type_ is OrderItemTariff.Type.PER_MINUTE:
            params_data = PerMinuteOrderItemTariffParamsSerializer(params).data
        else:
            LOGGER.error('unhandled order item tariff type: %s', type_)
            params_data = {}
        return params_data


class OrderItemPaymentSerializer(ModelSerializer):

    status = SerializerMethodField()
    impl = SerializerMethodField()

    class Meta:
        model = OrderItemPayment
        fields = [
            'created_at',
            'payment_method',
            'amount',
            'status',
            'impl',
        ]

    def get_status(self, obj):
        return obj.get_impl().get_generic_status().value

    def get_impl(self, obj):
        payment_method = obj.get_payment_method()
        if payment_method is OrderItemPayment.PaymentMethod.CARD:
            serializer_class = CardPaymentSerializer
        else:
            serializer_class = None

        if serializer_class is None:
            impl_data = None
        else:
            impl_data = serializer_class(obj.get_impl()).data

        return impl_data


class CardPaymentSerializer(ModelSerializer):

    fiscal_receipt_url = SerializerMethodField()

    class Meta:
        model = CardPayment
        fields = [
            'purchase_token',
            'status',
            'amount',
            'resp_code',
            'resp_desc',
            'fiscal_receipt_url',
        ]

    def get_fiscal_receipt_url(self, obj):
        if obj.purchase_token is None:
            url = None
        else:
            url = 'https://trust.yandex.ru/checks/{}/receipts/{}?mode=pdf'.format(
                obj.purchase_token,
                obj.purchase_token,
            )
        return url


class OrderItemSerializer(ModelSerializer):
    started_at = TimestampField()
    finished_at = TimestampField()

    carsharing_acceptance = CarsharingAcceptanceSerializer()
    carsharing_reservation = CarsharingReservationSerializer()
    carsharing_ride = CarsharingRideSerializer()
    carsharing_parking = CarsharingParkingSerializer()

    cost = SerializerMethodField()
    tariff = OrderItemTariffSerializer()
    payments = OrderItemPaymentSerializer(many=True)

    class Meta:
        model = OrderItem
        fields = [
            'id',
            'type',
            'started_at',
            'finished_at',
            'carsharing_acceptance',
            'carsharing_reservation',
            'carsharing_ride',
            'carsharing_parking',
            'cost',
            'tariff',
            'payments',
        ]

    def get_cost(self, obj):
        # order_payment_processor = self.context['order_payment_processor']
        return {
            'value': None  # float(order_payment_processor.get_order_item_cost(obj)),
        }


class OrderSerializer(ModelSerializer):
    created_at = TimestampField()
    completed_at = TimestampField()
    user = SimpleUserSerializer()
    items = SerializerMethodField()
    cost = SerializerMethodField()

    class Meta:
        model = Order
        fields = [
            'id',
            'created_at',
            'completed_at',
            'user',
            'items',
            'cost',
            'payment_status',
        ]

    def get_items(self, obj):
        return OrderItemSerializer(obj.get_sorted_items(), context=self.context, many=True).data

    def get_cost(self, obj):
        return {
            'value': None,
        }
        # order_payment_processor = self.context['order_payment_processor']
        # summary = float(order_payment_processor.get_order_cost(obj))
        # return {
        #     'value': summary,
        #     'plus': self.get_plus(obj),
        # }

    def get_plus(self, obj):
        if obj.has_plus_discount:
            order_payment_processor = self.context['order_payment_processor']
            summary = float(order_payment_processor.get_order_cost(obj))
            discount = cars.settings.CARSHARING['plus']['discount_multiplier']
            return {
                'is_plus_discounted': True,
                'discounted_value': round(float(summary) / float(discount) - float(summary), 2),
                'discount': int((1.0 - float(discount)) * 100 + 0.1),  # +0.1 to avoid int(0.58 * 100) effect
            }
        else:
            return {
                'is_plus_discounted': False,
                'discounted_value': 0.0,
                'discount': 0,
            }
