from rest_framework.serializers import CharField, IntegerField, ModelSerializer, SerializerMethodField, ChoiceField

import cars.settings

from cars.core.history import HistoryAction
from cars.django.serializers import BaseSerializer, PhoneNumberField, TimestampField

from cars.callcenter.serializers import RequestUserSerializer
from cars.callcenter.models.phone_blacklist import CallCenterBlacklistedPhoneHistory


class TelephonyRegisterCallArgumentsSerializer(BaseSerializer):
    from_number = PhoneNumberField(required=True)
    to_number = IntegerField(min_value=1, required=True)


class TelephonyAssignedCallArgumentsSerializer(BaseSerializer):
    phone = PhoneNumberField(required=False)  # no default as phone may be None


class TelephonyOperatorsArgumentsSerializer(BaseSerializer):
    department_url_prefix = CharField(max_length=100, required=False, default='')


class PriorityUserArgumentsSerializer(BaseSerializer):
    phone = PhoneNumberField(required=True)


class BlacklistModificationArgumentsSerializer(BaseSerializer):
    phone = PhoneNumberField(required=True)


class BlacklistAddPhoneArgumentsSerializer(BlacklistModificationArgumentsSerializer):
    blocked_until = TimestampField(required=False, default=None)

    comment = CharField(
        required=False,
        allow_blank=True,
        default='',
        max_length=4096
    )


class BlacklistHistoryArgumentsSerializer(BaseSerializer):
    phone = PhoneNumberField(required=False)
    since = TimestampField(required=False)
    until = TimestampField(required=False)
    limit = IntegerField(required=False, min_value=1, max_value=1000, default=50)


class BlacklistHistorySerializer(ModelSerializer):
    action = SerializerMethodField()
    performed_at = SerializerMethodField()
    performed_by = SerializerMethodField()
    phone = CharField()
    blocked_until = TimestampField()
    comment = CharField()

    class Meta:
        model = CallCenterBlacklistedPhoneHistory
        fields = [
            'action',
            'performed_at',
            'performed_by',
            'phone',
            'blocked_until',
            'comment',
        ]

    def get_action(self, obj):
        return HistoryAction(obj.history_action).name

    def get_performed_at(self, obj):
        return TimestampField().to_representation(obj.history_timestamp)

    def get_performed_by(self, obj):
        return RequestUserSerializer(obj.history_user).data


class LoadBalanceArgumentsSerializer(BaseSerializer):
    city = ChoiceField(choices=list(cars.settings.CALLCENTER['distribute_apps']), required=True)
