import operator

from rest_framework.serializers import (
    BooleanField, CharField, ChoiceField, IntegerField, ModelSerializer, SerializerMethodField, UUIDField
)

from cars.callcenter.core import CallPriorityManager, PhoneBlacklistManager
from cars.django.serializers import BaseSerializer, TimestampField
from cars.users.models import User
from cars.registration_yang.models import YangAssignment
from ..core.push_client import PUSH_CLIENT
from .yang_assignment import YangAssignmentSerializer
from cars.users.models import UserPhoneBinding


class UserCreateViewArgumentsSerializer(BaseSerializer):
    uid = IntegerField(required=True)


class UserPhoneHistoryViewArgumentsSerializer(BaseSerializer):
    user_id = UUIDField(required=True)


class UserPhoneHistoryViewSerializer(ModelSerializer):
    user_id = SerializerMethodField()
    phone = CharField()
    submit_date = TimestampField()
    commit_date = TimestampField()

    class Meta:
        model = UserPhoneBinding
        fields = [
            'user_id',
            'phone',
            'submit_date',
            'commit_date',
        ]

    def get_user_id(self, obj):
        return getattr(obj.user, 'id', None)


class UserListArgumentsSerializer(BaseSerializer):
    q = CharField(required=False)
    status = ChoiceField(choices=[x.value for x in User.Status], required=False)
    tag = CharField(required=False)
    chat_action = CharField(required=False)
    chat_completed = BooleanField(required=False)


class UserSerializer(ModelSerializer):
    call_priority_manager = CallPriorityManager.from_settings()
    phone_blacklist_manager = PhoneBlacklistManager.from_settings()

    registered_at = TimestampField()

    assignments = SerializerMethodField()

    blackbox_info = SerializerMethodField()
    call_priority = SerializerMethodField()
    phone_blacklisted = SerializerMethodField()

    uid = SerializerMethodField()
    username = SerializerMethodField()
    email = SerializerMethodField()
    phone = SerializerMethodField()
    is_phone_verified = SerializerMethodField()
    first_name = SerializerMethodField()
    last_name = SerializerMethodField()
    patronymic_name = SerializerMethodField()

    class Meta:
        model = User
        fields = [
            'id',
            'uid',
            'username',
            'email',
            'phone',
            'is_phone_verified',
            'status',
            'registered_at',
            'first_name',
            'last_name',
            'patronymic_name',
            'assignments',
            'blackbox_info',
            'call_priority',
            'phone_blacklisted',
        ]

    def get_assignments(self, obj):
        assignments = (
            YangAssignment.objects
            .filter(license_back__user=obj)
            .order_by('created_at')
        )
        result = [YangAssignmentSerializer(a, context=self.context).data for a in assignments]
        return result

    def get_call_priority(self, obj):
        if obj.status == 'deleted':
            return False
        return self.call_priority_manager.is_user_prioritized(user_id=obj.id)

    def get_phone_blacklisted(self, obj):
        if obj.status == 'deleted':
            return False
        if obj.phone is not None:
            is_blacklisted, _ = self.phone_blacklist_manager.is_phone_blacklisted(phone=obj.phone)
        else:
            is_blacklisted = False
        return is_blacklisted

    def get_plus(self, obj):
        if obj.status == 'deleted':
            return False
        return obj.get_plus_status()

    def get_blackbox_info(self, obj):
        if obj.status == 'deleted':
            return None

        blackbox_info = self.context.get('blackbox_info', None)  # already initialized cars.core.passport.AccountData

        if blackbox_info is None or blackbox_info.is_erroneous:
            return {}

        social_account_info = {'is_social': blackbox_info.is_social}
        social_account_info.update(blackbox_info.social_account_info or {})

        hosted_account_info = {
            'is_hosted': blackbox_info.is_hosted,
            'external_domain': blackbox_info.external_domain,
        }

        social_profiles_info = blackbox_info.social_profiles_info

        formatted_blackbox_info = {
            'exists': blackbox_info.exists,
            'plus_status': blackbox_info.plus_status,
            'public_name': blackbox_info.public_name,
            'social_account_info': social_account_info,
            'hosted_account_info': hosted_account_info,
            'social_profiles_info': social_profiles_info,
        }

        return formatted_blackbox_info

    def get_uid(self, obj):
        return obj.uid if obj.status != 'deleted' else None

    def get_is_yandexoid(self, obj):
        return obj.is_yandexoid if obj.status != 'deleted' else None

    def get_username(self, obj):
        return obj.username if obj.status != 'deleted' else None

    def get_email(self, obj):
        return obj.email if obj.status != 'deleted' else None

    def get_phone(self, obj):
        return str(obj.phone) if obj.status != 'deleted' and obj.phone else None

    def get_is_phone_verified(self, obj):
        return obj.is_phone_verified if obj.status != 'deleted' else None

    def get_first_name(self, obj):
        return obj.first_name if obj.status != 'deleted' else None

    def get_last_name(self, obj):
        return obj.last_name if obj.status != 'deleted' else None

    def get_patronymic_name(self, obj):
        return obj.patronymic_name if obj.status != 'deleted' else None


class UserChatProxyViewArgumentsSerializer(BaseSerializer):

    action = CharField(required=True)
    id = CharField(required=True)
    param = CharField(required=True)
