import logging
import operator

from rest_framework.reverse import reverse
from rest_framework.serializers import CharField, ModelSerializer, SerializerMethodField

from cars.django.serializers import BaseSerializer, TimestampField
from cars.users.models import UserDocument, UserDocumentPhoto
from .user_documents_textdata import DrivingLicenseTextDataSerializer, PassportTextDataSerializer


LOGGER = logging.getLogger(__name__)


class UserDocumentSerializer(BaseSerializer):

    def get_document_type(self):
        raise NotImplementedError

    def get_document_photos(self, obj, type_):
        photos_qs = UserDocumentPhoto.objects.filter(user=obj.user)
        photos = [
            p
            for p in photos_qs
            if p.type == type_.value
        ]
        photos = sorted(photos, key=operator.attrgetter('submitted_at'))
        return [UserDocumentPhotoSerializer(p, context=self.context).data for p in photos]


class UserDriverLicenseSerializer(UserDocumentSerializer):

    back_photos = SerializerMethodField()
    front_photos = SerializerMethodField()
    text_data = SerializerMethodField()

    def get_document_type(self):
        return UserDocument.Type.DRIVER_LICENSE

    def get_back_photos(self, obj):
        return self.get_document_photos(obj, UserDocumentPhoto.Type.DRIVER_LICENSE_BACK)

    def get_front_photos(self, obj):
        return self.get_document_photos(obj, UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT)

    def get_text_data(self, obj):
        try:
            license_raw_textdata = (
                self.context['datasync_client'].get_license_unverified(
                    obj.user.uid, obj.user.driving_license_ds_revision
                ) or {}
            )
        except Exception:
            license_raw_textdata = {}
            LOGGER.exception('Exception while getting license textdata, uid=%d', obj.user.uid)

        result = DrivingLicenseTextDataSerializer(license_raw_textdata).data
        result['raw'] = license_raw_textdata
        return result


class UserPassportSerializer(UserDocumentSerializer):

    biographical_photos = SerializerMethodField()
    face_photos = SerializerMethodField()
    registration_photos = SerializerMethodField()
    selfie_photos = SerializerMethodField()
    text_data = SerializerMethodField()

    def get_document_type(self):
        return UserDocument.Type.PASSPORT

    def get_biographical_photos(self, obj):
        return self.get_document_photos(obj, UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL)

    def get_face_photos(self, obj):
        return self.get_document_photos(obj, UserDocumentPhoto.Type.FACE)

    def get_registration_photos(self, obj):
        return self.get_document_photos(obj, UserDocumentPhoto.Type.PASSPORT_REGISTRATION)

    def get_selfie_photos(self, obj):
        return self.get_document_photos(obj, UserDocumentPhoto.Type.PASSPORT_SELFIE)

    def get_text_data(self, obj):
        try:
            passport_raw_textdata = (
                self.context['datasync_client'].get_passport_unverified(
                    obj.user.uid, obj.user.passport_ds_revision
                ) or {}
            )
        except Exception:
            passport_raw_textdata = {}
            LOGGER.exception('Exception while getting passport textdata, uid=%d', obj.user.uid)

        result = PassportTextDataSerializer(passport_raw_textdata).data
        result['raw'] = passport_raw_textdata
        return result


class UserDocumentPhotoSerializer(ModelSerializer):

    submitted_at = TimestampField()
    verified_at = TimestampField()
    verification_status = CharField()
    url = SerializerMethodField()
    background_video_url = SerializerMethodField()

    class Meta:
        model = UserDocumentPhoto
        fields = [
            'type',
            'submitted_at',
            'verified_at',
            'verification_status',
            'url',
            'background_video_url',
        ]

    def get_url(self, obj):
        return reverse(
            'cars-admin:user-document-photo-content',
            kwargs={
                'user_id': obj.user_id,
                'document_id': obj.document_id if obj.document_id else 'null',
                'photo_id': obj.id,
            },
            request=self.context['request'],
        )

    def get_background_video_url(self, obj):
        return reverse(
            'cars-admin:user-document-photo-background-video',
            kwargs={
                'user_id': obj.user_id,
                'document_id': obj.document_id if obj.document_id else 'null',
                'photo_id': obj.id,
            },
            request=self.context['request'],
        )
