from rest_framework.response import Response

from cars.core.authorization import DrivePermissionAPIView, DriveActionPermissionFactory
from cars.bans.core.ban_manager import BanManager
from cars.bans.models.ban import Ban
from cars.django.pagination import Pagination
from cars.users.models.user import User
from ..serializers.bans import BanSerializer, UserBlockArgumentsSerializer
from ..serializers.user import UserSerializer
from ..permissions import AdminPermissionCode, AdminPermissionFactory


class UserBanListView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build(('support_py_general', 'support_py_sup_general'), require_all=False)]

    lookup_url_kwarg = 'user_id'
    pagination_class = Pagination

    def get_queryset(self):
        return User.objects.all()

    def do_get(self, request, user_id):  # pylint: disable=unused-argument
        user = self.get_object()
        bans = (
            Ban.objects
            .select_related(
                'started_by__registration_state',
                'finished_by__registration_state',
            )
            .filter(user=user)
            .order_by('-started_at')
        )
        paginated_bans = self.paginate_queryset(bans)
        data = BanSerializer(paginated_bans, many=True).data
        response = self.get_paginated_response(data)
        return response


class BaseBanUserView(DrivePermissionAPIView):
    action_permission_classes = [DriveActionPermissionFactory.build('support_py_reg')]

    lookup_url_kwarg = 'user_id'

    ban_manager = BanManager.from_settings()

    def get_queryset(self):
        return User.objects.all()

    def do_post(self, request, user_id):  # pylint: disable=unused-argument
        user = self.get_object()

        error = None
        try:
            user = self._do(user=user, request=request)
        except self.ban_manager.Error as e:
            error = str(e)

        if error is None:
            data = {
                'status': 'success',
                'user': UserSerializer(user, context=self.get_serializer_context()).data,
            }
        else:
            data = {
                'status': 'errors',
                'errors': [error],
            }

        return Response(data)

    def _do(self, user, request):
        raise NotImplementedError


class BlockUserView(BaseBanUserView):

    arguments_serializer_class = UserBlockArgumentsSerializer

    def _do(self, user, request):
        user = self.ban_manager.block(
            user=user,
            operator=request.user,
            reason=request.arguments['reason'],
            comment=request.arguments.get('comment'),
        )
        return user


class UnblockUserView(BaseBanUserView):

    def _do(self, user, request):
        user = self.ban_manager.unblock(
            user=user,
            operator=request.user,
        )
        return user
