import logging

from rest_framework.exceptions import AuthenticationFailed, PermissionDenied

import cars.settings
from cars.django.permissions import IsAuthenticated
from cars.django.views import CarsharingAPIView
from cars.users.models import User, UserRole
from ..permissions import AdminPermissionCode, AdminPermissionFactory


LOGGER = logging.getLogger(__name__)


class AdminAPIView(CarsharingAPIView):

    acl = cars.settings.ADMIN['acl']

    access_permission_class = AdminPermissionFactory.build(AdminPermissionCode.ACCESS)
    permission_classes = [IsAuthenticated]

    def check_banned_roles(self, user):
        roles = (
            UserRole.objects
            .filter(
                user=user,
                role__role_id__in=[
                    'team_adm_level1',
                    'team_adm_level2',
                    'team_adm_level3'
                ],
                active=True,
            )
        )
        if roles.count():
            raise PermissionDenied

    def initial(self, request, *args, **kwargs):
        if request.user.id == '4c89fb49-036b-4297-98b9-77ddaac6068c':
            LOGGER.error('access from banned id')
            raise PermissionDenied
        if request.user.id == '40369ce9-8d63-40bd-93eb-f2ae741b4182':
            LOGGER.error('access from banned id')
            raise PermissionDenied

        if not self._check_acl(request.user):
            raise PermissionDenied
        return super().initial(request, *args, **kwargs)

    def _check_acl(self, user):
        if not user:
            raise AuthenticationFailed

        if self.acl is not None and user.uid in self.acl:
            return True

        if user.has_perm(self.access_permission_class.full_code):
            return True

        return False


class AdminAndCallcenterAPIView(AdminAPIView):

    internal_callcenter_acl = cars.settings.ADMIN['internal_callcenter_acl']

    def _check_acl(self, user):
        if super()._check_acl(user):
            return True

        if self.internal_callcenter_acl is None:
            return True
        return user.is_authenticated and user.uid in self.internal_callcenter_acl
